import moment, {Moment} from 'moment';

import {Request} from '@yandex-data-ui/core/lib/types';
import {IAviaParams} from '../types/IAviaParams';

import {ROBOT} from 'utilities/dateUtils/formats';

export function getAviaSearchFormData(req: Request): IAviaParams | undefined {
    if (isValid(req)) {
        return convert(req);
    }
}

function isValid(req: Partial<Request>): req is IAviaParams {
    if (!isValidDate(req.query.when)) {
        return false;
    }

    if (req.query.return_date && !isValidDate(req.query.return_date)) {
        return false;
    }

    /* eslint-disable camelcase */
    const {adult_seats, children_seats, infant_seats} = req.query;

    if (
        seatIsNotNumber(adult_seats) ||
        seatIsNotNumber(children_seats) ||
        seatIsNotNumber(infant_seats)
    ) {
        return false;
    }

    const {fromId, toId} = req.query;

    if (!fromId || !toId) {
        return false;
    }

    return true;
}

function isValidDate(dateString?: string): boolean {
    const momentDate = getMomentDate(dateString);

    return momentDate.isValid();
}

function getMomentDate(dateString?: string): Moment {
    return moment(dateString, ROBOT);
}

function seatIsNotNumber(seat: string): boolean {
    return isNaN(Number(seat));
}

function convert(req: Request): IAviaParams {
    /* eslint-disable camelcase */
    const {
        when,
        return_date,
        adult_seats,
        children_seats,
        infant_seats,
        fromId,
        toId,
        fromName,
        toName,
        klass,
        oneway,
    } = req.query;

    return {
        when: getMomentDate(when).format(ROBOT),
        return_date: return_date
            ? getMomentDate(return_date).format(ROBOT)
            : '',
        adult_seats,
        children_seats,
        infant_seats,
        fromId,
        toId,
        fromName,
        toName,
        klass,
        oneway,
    };
}
