import _set from 'lodash/set';

import {MAX_SEARCH_SUGGEST_COUNT} from 'constants/avia';

import {ESearchFormFieldName} from 'components/SearchForm/types';
import {
    EAviaSuggestsApiItemType,
    IAviaSuggestsApiItemData,
    TAviaGetSuggestsApiResponse,
    TAviaSuggestsApiItem,
} from 'server/api/AviaSuggestsApi/types/TAviaGetSuggestsApiResponse';
import {IAviaGetSuggestsServiceParams} from 'server/services/AviaService/types/IAviaGetSuggestsService';
import {isNotNull} from 'types/utilities';

import {getSuggestForAnywhere} from 'projects/avia/lib/getSuggestForAnywhere';

import * as i18nSearchToAnywhere from 'i18n/avia-SearchToAnywhere';
import * as i18nSearchForm from 'i18n/avia-SearchForm';

type TSuggestHash = {
    [key: number]: {
        [key: string]: number;
    };
};

type TOptions = {
    suggestHash: TSuggestHash;
    type: EAviaSuggestsApiItemType;
    title: string;
};

export interface IPreparedAviaSuggestItem {
    type: EAviaSuggestsApiItemType;
    /** @example 'Екатеринбург', 'Куда угодно' */
    title: string;
    /** @example 'SVX' */
    pointCode: string;
    /** @example 'c54' */
    pointKey: string;
    /** @example 'Россия' */
    countryTitle: string;
    /** @example 'Екатеринбург' */
    cityTitle: string;
    /** @example 'Свердловская область' */
    regionTitle: string;

    parentId?: boolean | string;
    /**
     * Описание для специального типа саджестов (EAviaSuggestsApiItemType.SPECIAL)
     * @example "Лучшие места для путешествия на ваши даты" для title 'Куда угодно'
     */
    specialDescription?: string;

    /** Имя группы */
    groupName?: string;
}

export interface IPreparedAviaSuggestItemWithIsUniqueTitle
    extends IPreparedAviaSuggestItem {
    isUniqueTitle: boolean;
}

export interface IPreparedAviaSuggestResponse {
    query: string;
    items: IPreparedAviaSuggestItemWithIsUniqueTitle[];
}

const prepareAviaSuggestItem = (
    geoOptions: IAviaSuggestsApiItemData,
    title: string,
    parentId: boolean | string,
    type: EAviaSuggestsApiItemType,
): IPreparedAviaSuggestItem => {
    const {
        point_code: pointCode,
        point_key: pointKey,
        country_title: countryTitle,
        city_title: cityTitle,
        region_title: regionTitle,
    } = geoOptions;

    return {
        type,
        title,
        parentId,
        pointCode: pointCode.toUpperCase(),
        pointKey,
        countryTitle,
        cityTitle,
        regionTitle,
    };
};

const isUniqueTitle = ({suggestHash, type, title}: TOptions): boolean =>
    suggestHash?.[type]?.[title] === 1;

const checkAndSetItemsUniqueTitle = (
    items: IPreparedAviaSuggestItem[] = [],
    suggestHash = {},
): IPreparedAviaSuggestItemWithIsUniqueTitle[] =>
    items.map(({type, title, ...restProps}) => ({
        ...restProps,
        title,
        type,
        isUniqueTitle: isUniqueTitle({suggestHash, type, title}),
    }));

const fillSuggestHash = ({
    suggestHash,
    type,
    title,
}: TOptions): TSuggestHash => {
    const countByTypeAndTitle = suggestHash?.[type]?.[title];
    const currentCount = countByTypeAndTitle ? countByTypeAndTitle + 1 : 1;
    const currentKey = `${type}.${title}`;

    return _set(suggestHash, currentKey, currentCount);
};

const normalizeAviaSuggestItems = (
    items: TAviaSuggestsApiItem[] = [],
    restItemsCount: number = 0,
    suggestHash: {} = {},
    parentId: boolean | string = false,
    showCountries: boolean = false,
): IPreparedAviaSuggestItem[] => {
    const restItems = items.slice(0, restItemsCount);
    let restChildrenCount = Math.max(0, restItemsCount - restItems.length);

    return restItems.reduce<IPreparedAviaSuggestItem[]>(
        (preparedItems, currentItem) => {
            const [type, title, geoOptions, children] = currentItem;
            const preparedItem =
                !showCountries && type === EAviaSuggestsApiItemType.COUNTRY
                    ? null
                    : prepareAviaSuggestItem(geoOptions, title, parentId, type);

            if (!preparedItem) {
                restChildrenCount++;
            }

            const childrenItems = normalizeAviaSuggestItems(
                children,
                restChildrenCount,
                suggestHash,
                preparedItem?.pointKey || false,
                showCountries,
            );

            fillSuggestHash({suggestHash, type, title});
            restChildrenCount -= childrenItems.length;

            return [...preparedItems, preparedItem, ...childrenItems].filter(
                isNotNull,
            );
        },
        [],
    );
};

const addSearchToAnywhere = (
    query: string,
    suggestItems: IPreparedAviaSuggestItemWithIsUniqueTitle[],
): IPreparedAviaSuggestItemWithIsUniqueTitle[] => {
    const needAddSuggestForAnywhere =
        query === '' ||
        i18nSearchToAnywhere
            .suggestTitle()
            .toLowerCase()
            .startsWith(query.toLowerCase());

    if (needAddSuggestForAnywhere) {
        const result = needAddSuggestForAnywhere
            ? [getSuggestForAnywhere()]
            : [];

        suggestItems.forEach(item => {
            result.push({
                ...item,
                groupName: i18nSearchForm.popularDirections(),
            });
        });

        return result;
    }

    return suggestItems;
};

export default function prepareSuggestsResponse(
    suggestRequest: IAviaGetSuggestsServiceParams,
    suggestResponse: TAviaGetSuggestsApiResponse,
): IPreparedAviaSuggestResponse {
    const {showCountries, showAnywhere, field} = suggestRequest;
    const [query, items] = suggestResponse;
    const suggestHash = {};
    const normalizedItems = normalizeAviaSuggestItems(
        items,
        MAX_SEARCH_SUGGEST_COUNT,
        suggestHash,
        false,
        showCountries,
    );
    let preparedItems = checkAndSetItemsUniqueTitle(
        normalizedItems,
        suggestHash,
    );

    if (showAnywhere && field === ESearchFormFieldName.TO) {
        preparedItems = addSearchToAnywhere(query, preparedItems);
    }

    return {
        query,
        items: preparedItems,
    };
}
