import {EFieldName} from 'components/BookingPassengerForm/constants/fieldNames';

import {IDocumentDTO} from 'server/api/TravelersApi/types/IDocumentDTO';
import {IBookingFormData} from 'projects/avia/pages/AviaBooking/types/IBookingFormData';
import {ITravelerNotebook} from 'projects/avia/pages/AviaBooking/types/ITravelerNotebook';
import {IPassengerWithDocumentsAndBonusCardsDTO} from 'server/api/TravelersApi/types/IPassengerDTO';
import {EGender} from 'types/common/document/EGender';
import {IBonusCardDTO} from 'server/api/TravelersApi/types/IBonusCardDTO';

import {ILogger} from 'server/utilities/Logger';
import {buildTravelerNotebook} from './utilities/buildTravelerNotebook';
import {parseDate} from 'utilities/dateUtils';
import {HUMAN_DATE_RU, ROBOT} from 'utilities/dateUtils/formats';
import {unknownToErrorOrUndefined} from 'utilities/error';

import {IDependencies} from 'server/getContainerConfig';
import {TravelersApi} from 'server/api/TravelersApi/TravelersApi';
import {EDocumentType} from 'server/api/TravelersApi/enums/EDocumentType';
import {EBonusCardType} from 'server/api/TravelersApi/enums/EBonusCardType';

export class AviaTravelersNotebookService {
    private travelersApi: TravelersApi;
    private uid: string | undefined;
    private logger: ILogger;

    constructor({travelersApi, userInfo, logger}: IDependencies) {
        this.travelersApi = travelersApi;
        this.logger = logger;
        this.uid = userInfo && 'uid' in userInfo ? userInfo.uid : undefined;
    }

    async getTravelersNotebook(): Promise<ITravelerNotebook | null> {
        if (!this.uid) {
            return null;
        }

        try {
            const [traveler, passengers] = await Promise.all([
                this.travelersApi.getTraveler(this.uid),
                this.travelersApi.getAllPassengers(this.uid, ['documents']),
            ]);

            /* В случае 404 ошибки от API вернется строка (html с 404) */
            if (typeof traveler === 'string') {
                throw new Error(traveler);
            }

            return buildTravelerNotebook(traveler, passengers);
        } catch (ex) {
            this.logger.logWarn(
                'getTravelersNotebook failure',
                unknownToErrorOrUndefined(ex),
            );

            return null;
        }
    }

    /**
     * @deprecated - использовать notebookService
     */
    async saveTravelerNotebookFromSubmit({
        passengers = [],
        contacts = {},
    }: IBookingFormData): Promise<IDocumentDTO[] | undefined> {
        const {email, phone} = contacts;
        const uid = this.uid;

        if (!uid || !phone || !email) {
            return;
        }

        try {
            await this.travelersApi.createOrUpdateTraveler(uid, {
                phone,
                email,
                agree: true,
            });
        } catch (ex) {
            this.logger.logWarn(
                'saveTravelerNotebookFromSubmit',
                unknownToErrorOrUndefined(ex),
            );
        }

        try {
            const apiPassengers = await this.travelersApi.getAllPassengers(
                uid,
                ['documents', 'bonus-cards'],
            );

            const passengersMutationPromises = passengers.map(
                async (passenger): Promise<void> => {
                    const passengerBirthdate =
                        passenger.birthdate &&
                        parseDate(passenger.birthdate, HUMAN_DATE_RU).format(
                            ROBOT,
                        );
                    const passengerGender = passenger[EFieldName.sex];
                    const passengerFirstName = passenger[EFieldName.firstName];
                    const passengerLastName = passenger[EFieldName.lastName];
                    const passengerPatronymic =
                        passenger[EFieldName.patronymicName];
                    const passengerDocumentType =
                        passenger[EFieldName.documentType];
                    const passengerDocumentNumber =
                        passenger[EFieldName.documentNumber];
                    const passengerDocumentCitizenship =
                        passenger.citizenshipGeoId;
                    const passengerDocumentExpirationDate =
                        passenger.documentValidDate
                            ? parseDate(
                                  passenger.documentValidDate,
                                  HUMAN_DATE_RU,
                              ).format(ROBOT)
                            : undefined;
                    const passengerLoyaltyCardType =
                        passenger.loyaltyCard?.type;
                    const passengerLoyaltyCardNumber =
                        passenger.loyaltyCard?.number;

                    if (
                        !passengerBirthdate ||
                        !passengerGender ||
                        !passengerDocumentType ||
                        !passengerDocumentNumber ||
                        !passengerFirstName ||
                        !passengerLastName
                    ) {
                        return;
                    }

                    const existedApiPassenger =
                        await this.findOrCreateApiPassenger(
                            apiPassengers,
                            uid,
                            {
                                gender: passengerGender,
                                birthdate: passengerBirthdate,
                                lastName: passengerLastName,
                                firstName: passengerFirstName,
                                patronymic: passengerPatronymic,
                            },
                        );

                    await this.findOrCreateApiDocument(
                        existedApiPassenger,
                        uid,
                        {
                            documentType: passengerDocumentType,
                            documentNumber: passengerDocumentNumber,
                            documentCitizenship: passengerDocumentCitizenship,
                            documentExpirationDate:
                                passengerDocumentExpirationDate,
                            passengerFirstName,
                            passengerLastName,
                            passengerPatronymic,
                        },
                    );

                    if (
                        !passengerLoyaltyCardType ||
                        !passengerLoyaltyCardNumber
                    ) {
                        return;
                    }

                    // Пока зкп не поддерживает сохранение карт лояльности авиа (https://st.yandex-team.ru/RASPTICKETS-19944)
                    // await this.findOrCreateApiLoyaltyCard(existedApiPassenger, uid, {
                    //     loyaltyCardType: passengerLoyaltyCardType,
                    //     loyaltyCardNumber: passengerLoyaltyCardNumber,
                    // });
                },
            );

            await Promise.all(passengersMutationPromises);
        } catch (err) {
            this.logger.logWarn(
                'saveTravelerNotebookFromSubmit',
                unknownToErrorOrUndefined(err),
            );
        }
    }

    private async findOrCreateApiPassenger(
        apiPassengers: IPassengerWithDocumentsAndBonusCardsDTO[],
        userUid: string,
        {
            gender,
            birthdate,
            lastName,
            firstName,
            patronymic,
        }: {
            gender: EGender;
            birthdate: string;
            lastName: string;
            firstName: string;
            patronymic: string | undefined;
        },
    ): Promise<IPassengerWithDocumentsAndBonusCardsDTO> {
        const existedApiPassenger = apiPassengers.find(apiPassenger => {
            const hasSamePersonApiPassenger =
                apiPassenger.gender === gender &&
                apiPassenger.birth_date === birthdate;

            const hasSameFIOApiDocument = apiPassenger.documents.some(
                apiDocument =>
                    apiDocument.last_name_en === lastName &&
                    apiDocument.first_name_en === firstName &&
                    (!patronymic || apiDocument.middle_name_en === patronymic),
            );

            return hasSamePersonApiPassenger && hasSameFIOApiDocument;
        });

        if (existedApiPassenger) {
            return existedApiPassenger;
        }

        const newApiPassenger = await this.travelersApi.createPassenger(
            userUid,
            {
                birth_date: birthdate,
                gender,
                title: `${lastName} ${firstName}${
                    patronymic ? ` ${patronymic}` : ''
                }`,
            },
        );

        return {
            ...newApiPassenger,
            documents: [],
            bonus_cards: [],
            itn: null,
        };
    }

    private async findOrCreateApiDocument(
        apiPassenger: IPassengerWithDocumentsAndBonusCardsDTO,
        userUid: string,
        {
            documentType,
            documentNumber,
            documentCitizenship,
            documentExpirationDate,
            passengerFirstName,
            passengerLastName,
            passengerPatronymic,
        }: {
            documentType: EDocumentType;
            documentNumber: string;
            documentCitizenship: number | undefined;
            documentExpirationDate: string | undefined;
            passengerFirstName: string;
            passengerLastName: string;
            passengerPatronymic: string | undefined;
        },
    ): Promise<IDocumentDTO> {
        const existedApiDocument = apiPassenger.documents.find(
            apiDocument =>
                apiDocument.type === documentType &&
                apiDocument.number === documentNumber &&
                apiDocument.citizenship === documentCitizenship,
        );

        if (existedApiDocument) {
            return existedApiDocument;
        }

        return this.travelersApi.createDocument(userUid, apiPassenger.id, {
            citizenship: documentCitizenship,
            number: documentNumber,
            type: documentType,
            first_name_en: passengerFirstName,
            last_name_en: passengerLastName,
            middle_name_en: passengerPatronymic || undefined,
            expiration_date: documentExpirationDate,
        });
    }

    // @ts-ignore
    private async findOrCreateApiLoyaltyCard(
        apiPassenger: IPassengerWithDocumentsAndBonusCardsDTO,
        userUid: string,
        {
            loyaltyCardType,
            loyaltyCardNumber,
        }: {
            loyaltyCardType: EBonusCardType;
            loyaltyCardNumber: string;
        },
    ): Promise<IBonusCardDTO> {
        const existedApiLoyaltyCard = apiPassenger.bonus_cards.find(
            apiLoyaltyCard =>
                apiLoyaltyCard.type === loyaltyCardType &&
                apiLoyaltyCard.number === loyaltyCardNumber,
        );

        if (existedApiLoyaltyCard) {
            return existedApiLoyaltyCard;
        }

        return this.travelersApi.createBonusCard(userUid, apiPassenger.id, {
            type: loyaltyCardType,
            number: loyaltyCardNumber,
            title: loyaltyCardType,
        });
    }
}
