import {IAviaTDResultsQuery} from 'server/api/AviaTicketDaemonApi/types/IAviaTDResultsQuery';
import {
    IAviaTDAnswer,
    IFare,
} from 'server/api/AviaTicketDaemonApi/types/IAviaTDAnswer';
import {IAviaTDApiOrderQuery} from 'server/api/AviaTicketDaemonApi/types/IAviaTDApiOrderQuery';
import {IAviaInitByQueryService} from 'server/services/AviaVariantsService/types/IAviaInitByQueryService';
import {ICookies} from '@yandex-data-ui/core/lib/types';

import {parseAviaDate} from 'projects/avia/lib/date/parseDate';
import {getAviaTestContext} from 'server/utilities/avia/getAviaTestContext';

import {
    AviaPartnersApi,
    TAviaPartnersMap,
} from 'server/api/AviaPartnersApi/AviaPartnersApi';
import {AviaTicketDaemonApi} from 'server/api/AviaTicketDaemonApi/AviaTicketDaemonApi';
import {IDependencies} from 'server/getContainerConfig';

import {AviaFlightStorageService} from '../AviaFlightStorageService';

export interface IAviaVariantsService {
    initByQuery: IAviaInitByQueryService;
}

export class AviaVariantsService {
    private aviaTicketDaemonApi: AviaTicketDaemonApi;
    private aviaPartnersApi: AviaPartnersApi;
    private aviaFlightStorageService: AviaFlightStorageService;
    private readonly cookies: ICookies;

    constructor({
        aviaTicketDaemonApi,
        aviaPartnersApi,
        aviaFlightStorageService,
        cookies,
    }: IDependencies) {
        this.aviaTicketDaemonApi = aviaTicketDaemonApi;
        this.aviaPartnersApi = aviaPartnersApi;
        this.aviaFlightStorageService = aviaFlightStorageService;
        this.cookies = cookies;
    }

    initByQuery: IAviaInitByQueryService = async searchParams => {
        return this.aviaTicketDaemonApi.initByQuery({
            adults: searchParams.adult_seats,
            children: searchParams.children_seats,
            infants: searchParams.infant_seats,
            date_backward: parseAviaDate(searchParams.return_date),
            date_forward: parseAviaDate(searchParams.when),
            klass: searchParams.klass,
            lang: 'ru',
            national: 'ru',
            point_from: searchParams.fromId,
            point_to: searchParams.toId,
            test_context: getAviaTestContext({
                cookies: this.cookies,
            }),
        });
    };

    async results(
        resultsQuery: IAviaTDResultsQuery,
        nationalVersionForPartnersDisabling: Nullable<string>,
    ): Promise<IAviaTDAnswer> {
        const [tdAnswer, partners] = await Promise.all([
            this.aviaTicketDaemonApi.results({
                ...resultsQuery,
                test_context: getAviaTestContext({
                    cookies: this.cookies,
                }),
            }),
            nationalVersionForPartnersDisabling
                ? this.aviaPartnersApi.getAll()
                : null,
        ]);

        if (nationalVersionForPartnersDisabling && partners) {
            return this.filterNonWizardPartners(
                partners,
                tdAnswer,
                nationalVersionForPartnersDisabling,
            );
        }

        return tdAnswer;
    }

    async variants(
        orderQuery: IAviaTDApiOrderQuery,
        nationalVersionForPartnersDisabling: Nullable<string>,
    ): Promise<IAviaTDAnswer> {
        const [tdAnswer, partners] = await Promise.all([
            this.aviaTicketDaemonApi.getOrder({
                ...orderQuery,
                test_context: getAviaTestContext({
                    cookies: this.cookies,
                }),
            }),
            nationalVersionForPartnersDisabling
                ? this.aviaPartnersApi.getAll()
                : null,
        ]);

        if (nationalVersionForPartnersDisabling && partners) {
            return this.filterNonWizardPartners(
                partners,
                tdAnswer,
                nationalVersionForPartnersDisabling,
            );
        }

        return tdAnswer;
    }

    async order(
        orderQuery: IAviaTDApiOrderQuery,
        nationalVersionForPartnersDisabling: Nullable<string>,
    ): Promise<IAviaTDAnswer | null> {
        const data = await this.variants(
            orderQuery,
            nationalVersionForPartnersDisabling,
        );

        if (data.variants.fares.length) {
            return data;
        }

        return this.flightStorageOrderFallback(orderQuery);
    }

    private async flightStorageOrderFallback(
        orderQuery: IAviaTDApiOrderQuery,
    ): Promise<IAviaTDAnswer | null> {
        return this.aviaFlightStorageService.getOrder({
            forward: orderQuery.forward,
            backward: orderQuery.backward,
        });
    }

    private filterNonWizardPartners(
        partners: TAviaPartnersMap,
        data: IAviaTDAnswer,
        nationalVersionForPartnersDisabling: string,
    ): IAviaTDAnswer {
        const filteredFares = data.variants.fares
            .map(fare => {
                const filteredPrices = fare.prices.filter(price => {
                    const partner = partners.get(price.partnerCode);

                    return (
                        !partner ||
                        partner.enabledInWizard[
                            nationalVersionForPartnersDisabling
                        ] !== false
                    );
                });

                return {
                    ...fare,
                    prices: filteredPrices,
                };
            })
            .filter(fare => fare.prices.length > 0) as IFare[];

        return {
            ...data,
            variants: {
                ...data.variants,
                fares: filteredFares,
            },
        };
    }
}
