import NodeCache from 'node-cache';

import {ENodeCacheKey} from 'server/constants/eNodeCacheKey';

import {TFeatureFlags} from 'types/IFeatureFlags';
import {ITrainPopularRoutesResponse} from 'server/api/BunkerApi/types/trainPopularRoutes';

import {prepareBunkerFeatureFlagsResponse} from './utilities/prepareBunkerFeatureFlagsResponse';

import {IDependencies} from 'server/getContainerConfig';
import {BunkerApi} from 'server/api/BunkerApi/BunkerApi';

export interface IBunkerService {
    getTrainPopularRoutes: () => Promise<ITrainPopularRoutesResponse>;
    getFeatureFlags: () => Promise<TFeatureFlags>;
}

const cache = new NodeCache({stdTTL: 20});

export class BunkerService implements IBunkerService {
    private bunkerApi: BunkerApi;

    constructor({bunkerApi}: IDependencies) {
        this.bunkerApi = bunkerApi;
    }

    getTrainPopularRoutes(): Promise<ITrainPopularRoutesResponse> {
        return this.bunkerApi.getTrainPopularRoutes();
    }

    async getFeatureFlags(): Promise<TFeatureFlags> {
        const dataFromCache = cache.get<TFeatureFlags | undefined>(
            ENodeCacheKey.BUNKER_FEATURE_FLAGS,
        );

        if (dataFromCache) {
            return dataFromCache;
        }

        const featureFlags = await this.bunkerApi
            .getFeatureFlags()
            .then(prepareBunkerFeatureFlagsResponse);

        cache.set(ENodeCacheKey.BUNKER_FEATURE_FLAGS, featureFlags);

        return featureFlags;
    }
}
