import {STATUS_AND_CODE_MAP} from 'projects/buses/constants/statuses/searchBooking';

import {IBusesGetSuggestsService} from 'server/services/BusesService/types/IBusesGetSuggestsService';
import {IBusesGetSegmentsService} from 'server/services/BusesService/types/IBusesGetSegmentsService';
import {IBusesGetGeoPointsService} from 'server/services/BusesService/types/IBusesGetGeoPointsService';
import {IBusesGetPopularDirectionsService} from 'server/services/BusesService/types/IBusesGetPopularDirectionsService';
import {IBusesGetRideService} from 'server/services/BusesService/types/IBusesGetRideService';
import {IBusesGetCalendarService} from 'server/services/BusesService/types/IBusesGetCalendarService';
import {IBusesCreateRideOfferService} from 'server/services/BusesService/types/IBusesCreateRideOfferService';
import {IBusesGetTestContextService} from 'server/services/BusesService/types/IBusesGetTestContextService';
import IAttributionData from 'server/utilities/DataStorage/AttributionData/types/IAttributionData';
import IUserData from 'server/utilities/DataStorage/UserData/types/IUserData';
import {IBusesGetDirection} from './types/IBusesGetDirection';

import AffiliateData from 'server/utilities/DataStorage/AffiliateData/AffiliateData';
import getLabelParams from 'server/services/BusesService/utilities/getLabelParams';

import {IDependencies} from 'server/getContainerConfig';
import {BusesGeoApi} from 'server/api/BusesGeoApi/BusesGeoApi';
import getPreparedGetSuggestsServiceResponse from 'server/services/BusesService/dataTransformers/getPreparedGetSuggestsServiceResponse';
import {BusesApi} from 'server/api/BusesApi/BusesApi';
import getPreparedGetSegmentsApiParams from 'server/services/BusesService/dataTransformers/getPreparedGetSegmentsApiParams';
import getPreparedGetSegmentsServiceResponse from 'server/services/BusesService/dataTransformers/getPreparedGetSegmentsServiceResponse';
import getPreparedGetPopularApiParams from 'server/services/BusesService/dataTransformers/getPreparedGetPopularApiParams';
import getPreparedGetPopularServiceResponse from 'server/services/BusesService/dataTransformers/getPreparedGetPopularServiceResponse';
import getPreparedGetCalendarApiParams from 'server/services/BusesService/dataTransformers/getPreparedGetCalendarApiParams';
import {BusesTravelApi} from 'server/api/BusesTravelApi/BusesTravelApi';
import {GeobaseService} from 'server/services/GeobaseService/GeobaseService';

export interface IBusesService {
    getSuggests: IBusesGetSuggestsService;
    getGeoPoints: IBusesGetGeoPointsService;
    getSegments: IBusesGetSegmentsService;
    getPopularDirections: IBusesGetPopularDirectionsService;
    getCalendar: IBusesGetCalendarService;
    getRide: IBusesGetRideService;
    getDirection: IBusesGetDirection;
    getTestContext: IBusesGetTestContextService;
}

export class BusesService implements IBusesService {
    private busesApi: BusesApi;
    private busesGeoApi: BusesGeoApi;
    private busesTravelApi: BusesTravelApi;
    private readonly geobaseService: GeobaseService;
    private readonly ip: string;
    private readonly attributionData: IAttributionData;
    private readonly userData: IUserData;
    private readonly affiliateData: AffiliateData;

    constructor({
        busesGeoApi,
        busesApi,
        busesTravelApi,
        geobaseService,
        ip,
        attributionData,
        userData,
        affiliateData,
    }: IDependencies) {
        this.busesApi = busesApi;
        this.busesGeoApi = busesGeoApi;
        this.busesTravelApi = busesTravelApi;
        this.geobaseService = geobaseService;
        this.ip = ip;
        this.attributionData = attributionData;
        this.userData = userData;
        this.affiliateData = affiliateData;
    }

    getSuggests: IBusesGetSuggestsService = async params => {
        const apiResponse = await this.busesGeoApi.getSuggests(params);

        return getPreparedGetSuggestsServiceResponse(apiResponse);
    };

    getGeoPoints: IBusesGetGeoPointsService = async params => {
        return this.busesGeoApi.getGeoPoints(params);
    };

    getPopularDirections: IBusesGetPopularDirectionsService = async params => {
        const preparedParams = getPreparedGetPopularApiParams(params);

        const apiResponse = await this.busesApi.getPopularDirections(
            preparedParams,
        );

        return getPreparedGetPopularServiceResponse(apiResponse);
    };

    getSegments: IBusesGetSegmentsService = async params => {
        const preparedParams = getPreparedGetSegmentsApiParams(params);

        const apiResponse = await this.busesApi.getSegments(preparedParams);

        return getPreparedGetSegmentsServiceResponse(
            apiResponse,
            params.fromTimezone,
            params.toTimezone,
        );
    };

    getCalendar: IBusesGetCalendarService = async params => {
        const preparedParams = getPreparedGetCalendarApiParams(params);

        return this.busesApi.getCalendar(preparedParams);
    };

    getRide: IBusesGetRideService = async params => {
        if (params.testStatus) {
            throw {
                response: {
                    status: STATUS_AND_CODE_MAP[params.testStatus],
                    data: 'Test error',
                },
            };
        }

        return this.busesApi.getRide(params);
    };

    createRideOffer: IBusesCreateRideOfferService = async params => {
        const {wizardReqId, ytpReferer, ...apiParams} = params;

        const labelParams = await getLabelParams({
            attributionData: this.attributionData,
            userData: this.userData,
            geobaseService: this.geobaseService,
            userIp: this.ip,
            affiliateParams: this.affiliateData.getParams(),
            wizardReqId,
            ytpReferer,
        });

        return this.busesTravelApi.createRideOffer({
            ...apiParams,
            labelParams,
        });
    };

    getDirection: IBusesGetDirection = async params => {
        return this.busesTravelApi.getBusesDirection(params);
    };

    getTestContext: IBusesGetTestContextService = async params => {
        return this.busesTravelApi.getTestContext(params);
    };
}
