import IGenericOrderService from 'server/services/GenericOrderService/types/IGenericOrderService';
import IGenericOrderCreateOrderApiResponse from 'server/api/GenericOrderApi/types/createOrder/IGenericOrderCreateOrderApiResponse';
import IGenericOrderAddServiceApiParams from 'server/api/GenericOrderApi/types/addService/IGenericOrderAddServiceApiParams';
import IGenericOrderAddServiceApiResponse from 'server/api/GenericOrderApi/types/addService/IGenericOrderAddServiceApiResponse';
import IGenericOrderCalculateRefundAmountApiParams from 'server/api/GenericOrderApi/types/calculateRefundAmount/IGenericOrderCalculateRefundAmountApiParams';
import IGenericOrderCalculateRefundAmountApiResponse from 'server/api/GenericOrderApi/types/calculateRefundAmount/IGenericOrderCalculateRefundAmountApiResponse';
import IGenericOrderCancelOrderApiParams from 'server/api/GenericOrderApi/types/cancelOrder/IGenericOrderCancelOrderApiParams';
import IGenericOrderEstimateDiscountApiParams from 'server/api/GenericOrderApi/types/estimateDiscount/IGenericOrderEstimateDiscountApiParams';
import IGenericOrderEstimateDiscountApiResponse from 'server/api/GenericOrderApi/types/estimateDiscount/IGenericOrderEstimateDiscountApiResponse';
import IGenericOrderGetOrderApiParams from 'server/api/GenericOrderApi/types/getOrder/IGenericOrderGetOrderApiParams';
import IGenericOrderGetOrderStateApiResponse from 'server/api/GenericOrderApi/types/getOrderState/IGenericOrderGetOrderStateApiResponse';
import IGenericOrderGetOrderStateApiParams from 'server/api/GenericOrderApi/types/getOrderState/IGenericOrderGetOrderStateApiParams';
import IGenericOrderStartPaymentApiParams from 'server/api/GenericOrderApi/types/startPayment/IGenericOrderStartPaymentApiParams';
import IGenericOrderStartRefundApiParams from 'server/api/GenericOrderApi/types/startRefund/IGenericOrderStartRefundApiParams';
import IGenericOrderCreateOrderServiceParams from 'server/services/GenericOrderService/types/createOrder/IGenericOrderCreateOrderServiceParams';
import {Request} from '@yandex-data-ui/core/lib/types';
import {Platform} from 'types/Platform';
import IGenericOrderGetOrderApiResponse from 'server/api/GenericOrderApi/types/getOrder/IGenericOrderGetOrderApiResponse';
import IUserData from 'server/utilities/DataStorage/UserData/types/IUserData';
import EGenericOrderSource from 'server/api/GenericOrderApi/types/common/EGenericOrderSource';
import IGenericOrderDownloadBlankApiParams from 'server/api/GenericOrderApi/types/downloadBlank/IGenericOrderDownloadBlankApiParams';

import {GenericOrderApi} from 'server/api/GenericOrderApi/GenericOrderApi';
import {IDependencies} from 'server/getContainerConfig';
import {TestContextService} from 'server/services/TestContextService/TestContextService';

import {IGeobaseService} from '../GeobaseService/GeobaseService';

export class GenericOrderService implements IGenericOrderService {
    private readonly userAgent: string | undefined;
    private readonly genericOrderApi: GenericOrderApi;
    private readonly userData: IUserData;
    private readonly testContextService: TestContextService;
    private readonly geobaseService: IGeobaseService;

    constructor({
        genericOrderApi,
        userAgent,
        userData,
        testContextService,
        geobaseService,
    }: IDependencies) {
        this.userData = userData;
        this.userAgent = userAgent;

        this.geobaseService = geobaseService;
        this.genericOrderApi = genericOrderApi;
        this.testContextService = testContextService;
    }

    async createOrder(
        params: IGenericOrderCreateOrderServiceParams,
        req: Request,
    ): Promise<IGenericOrderCreateOrderApiResponse> {
        if (!req) {
            throw new Error('req is undefined');
        }

        const {trainServices} = params;
        const {geoId} = await this.geobaseService.getUserGeoLocation();
        const paymentTestContextToken =
            await this.testContextService.getPaymentTestContextTokenIfNeeded(
                params.paymentTestContextToken,
            );
        const trainServicesWithTestTokenIfNeed =
            await this.testContextService.addTestContextTokenToTrainServicesIfNeed(
                trainServices,
            );

        return this.genericOrderApi.createOrder({
            ...params,
            trainServices: trainServicesWithTestTokenIfNeed,
            userInfo: {
                ip: req.ip,
                geoId: geoId,
                userAgent: this.userAgent,
                mobile: this.userData.getDeviceType() === Platform.TOUCH,
            },
            paymentTestContextToken,
        });
    }

    addService(
        params: IGenericOrderAddServiceApiParams,
    ): Promise<IGenericOrderAddServiceApiResponse> {
        return this.genericOrderApi.addService(params);
    }

    cancelOrder(params: IGenericOrderCancelOrderApiParams): Promise<void> {
        return this.genericOrderApi.cancelOrder(params);
    }

    estimateDiscount(
        params: IGenericOrderEstimateDiscountApiParams,
    ): Promise<IGenericOrderEstimateDiscountApiResponse> {
        return this.genericOrderApi.estimateDiscount(params);
    }

    getOrder({
        source = EGenericOrderSource.OTHER,
        ...rest
    }: IGenericOrderGetOrderApiParams): Promise<IGenericOrderGetOrderApiResponse> {
        return this.genericOrderApi.getOrder({source, ...rest});
    }

    getOrderState(
        params: IGenericOrderGetOrderStateApiParams,
    ): Promise<IGenericOrderGetOrderStateApiResponse> {
        return this.genericOrderApi.getOrderState(params);
    }

    startPayment(params: IGenericOrderStartPaymentApiParams): Promise<void> {
        return this.genericOrderApi.startPayment(params);
    }

    calculateRefundAmount(
        params: IGenericOrderCalculateRefundAmountApiParams,
    ): Promise<IGenericOrderCalculateRefundAmountApiResponse> {
        return this.genericOrderApi.calculateRefundAmount(params);
    }

    startRefund(params: IGenericOrderStartRefundApiParams): Promise<void> {
        return this.genericOrderApi.startRefund(params);
    }

    downloadBlank(
        params: IGenericOrderDownloadBlankApiParams,
    ): Promise<NodeJS.ReadableStream> {
        return this.genericOrderApi.downloadBlank(params);
    }
}
