/* eslint-disable @typescript-eslint/naming-convention */
import {
    ILegacyHotelInfoResponse,
    ILegacyHotelInfoRequestParams,
} from 'server/api/HotelSearchAPI/types/ILegacyHotelInfo';
import {
    IHotelInfo,
    IHotelInfoRequestParams,
} from 'server/api/HotelSearchAPI/types/IHotelInfo';
import {
    IHotelImages,
    IHotelImagesRequestParams,
} from 'server/api/HotelSearchAPI/types/IHotelImages';
import {
    IHotelReviewsInfo,
    IHotelReviewsInfoRequestParams,
} from 'server/api/HotelSearchAPI/types/IHotelReviews';
import {
    IAddHotelReviewRequest,
    IAddHotelReviewResponse,
} from 'server/api/HotelSearchAPI/types/IAddHotelReview';
import {
    IEditHotelReviewRequest,
    IEditHotelReviewResponse,
} from 'server/api/HotelSearchAPI/types/IEditHotelReview';
import {
    IHotelOffersInfo,
    IHotelOffersInfoRequestParams,
} from 'server/api/HotelSearchAPI/types/IOffersInfo';
import {
    IHotelReviewReactionResponse,
    IHotelReviewReactionRequestParams,
} from 'server/api/HotelSearchAPI/types/IHotelReviewReaction';
import {
    ISimilarHotelsInfo,
    ISimilarHotelsInfoRequestParams,
} from 'server/api/HotelSearchAPI/types/ISimilarHotels';
import {
    IDeleteHotelReviewRequest,
    IDeleteHotelReviewResponse,
} from 'server/api/HotelSearchAPI/types/IDeleteHotelReview';
import {Request} from '@yandex-data-ui/core/lib/types';
import {
    IUploadHotelReviewImageRequest,
    IUploadHotelReviewImageResponse,
} from 'server/api/HotelSearchAPI/types/IUploadHotelReviewImage';
import {
    IDeleteHotelReviewImagesRequestQuery,
    IDeleteHotelReviewImagesResponse,
    IDeleteHotelReviewImagesRequestBody,
} from 'server/api/HotelSearchAPI/types/IDeleteHotelReviewImages';
import {
    ICalendarPricesRequest,
    ICalendarPricesResponse,
} from 'server/api/HotelSearchAPI/types/ICalendarPrices';

import {prepareHotelReviews} from 'server/api/HotelSearchAPI/utilities/prepareHotelReviews';
import {prepareHotelOffersInfo} from 'server/api/HotelSearchAPI/utilities/prepareHotelOffersInfo';
import {prepareSimilarHotelsInfo} from 'server/api/HotelSearchAPI/utilities/prepareSimilarHotelsInfo';
import {HotelsMetaParamsBuilder} from 'server/utilities/hotels/HotelsMetaParamsBuilder/HotelsMetaParamsBuilder';

import {HotelSearchAPI} from 'server/api/HotelSearchAPI/HotelSearchAPI';

import {IDependencies} from '../../getContainerConfig';

export type GetLegacyHotelInfoType = (
    searchParams: ILegacyHotelInfoRequestParams,
) => Promise<ILegacyHotelInfoResponse>;

export type GetHotelInfoType = (
    searchParams: IHotelInfoRequestParams,
) => Promise<IHotelInfo>;

export type GetHotelImagesType = (
    imagesParams: IHotelImagesRequestParams,
) => Promise<IHotelImages>;

export type GetHotelReviewsType = (
    reviewsParams: IHotelReviewsInfoRequestParams,
) => Promise<IHotelReviewsInfo>;

export type GetHotelOffersInfoType = (
    offersParams: IHotelOffersInfoRequestParams,
) => Promise<IHotelOffersInfo>;

export type GetSimilarHotelsType = (
    similarHotelsParams: ISimilarHotelsInfoRequestParams,
) => Promise<ISimilarHotelsInfo>;

export type SetHotelReviewReactionType = (
    reaction: IHotelReviewReactionRequestParams,
) => Promise<IHotelReviewReactionResponse>;

export type AddHotelReviewType = (
    params: IAddHotelReviewRequest,
) => Promise<IAddHotelReviewResponse>;

export type EditHotelReviewType = (
    params: IEditHotelReviewRequest,
) => Promise<IEditHotelReviewResponse>;

export type DeleteHotelReviewType = (
    params: IDeleteHotelReviewRequest,
) => Promise<IDeleteHotelReviewResponse>;

export type UploadHotelReviewImageType = (
    formData: FormData,
    request: IUploadHotelReviewImageRequest,
) => Promise<IUploadHotelReviewImageResponse>;

export type ProxyUploadHotelReviewImageType = (
    req: Request,
) => Promise<IUploadHotelReviewImageResponse>;

export type DeleteHotelReviewImagesType = (
    params: IDeleteHotelReviewImagesRequestQuery,
    request: IDeleteHotelReviewImagesRequestBody,
) => Promise<IDeleteHotelReviewImagesResponse>;

export type GetHotelCalendarPricesType = (
    request: ICalendarPricesRequest,
) => Promise<ICalendarPricesResponse>;

export interface IHotelSearchService {
    getLegacyHotelInfo: GetLegacyHotelInfoType;
    getHotelInfo: GetHotelInfoType;
    getHotelImages: GetHotelImagesType;
    getHotelReviews: GetHotelReviewsType;
    addHotelReview: AddHotelReviewType;
    editHotelReview: EditHotelReviewType;
    deleteHotelReview: DeleteHotelReviewType;
    getHotelOffers: GetHotelOffersInfoType;
    getSimilarHotels: GetSimilarHotelsType;
    setHotelReviewReaction: SetHotelReviewReactionType;
    uploadHotelReviewImage: UploadHotelReviewImageType;
    proxyUploadHotelReviewImage: ProxyUploadHotelReviewImageType;
    deleteHotelReviewImages: DeleteHotelReviewImagesType;
    getCalendarPrices: GetHotelCalendarPricesType;
}

export class HotelSearchService implements IHotelSearchService {
    private hotelSearchAPI: HotelSearchAPI;
    private hotelsMetaParamsBuilder: HotelsMetaParamsBuilder;

    constructor({hotelSearchAPI, hotelsMetaParamsBuilder}: IDependencies) {
        this.hotelSearchAPI = hotelSearchAPI;
        this.hotelsMetaParamsBuilder = hotelsMetaParamsBuilder;
    }

    async getLegacyHotelInfo(
        searchParams: ILegacyHotelInfoRequestParams,
    ): Promise<ILegacyHotelInfoResponse> {
        return this.hotelSearchAPI.getLegacyHotelInfo(searchParams);
    }

    async getHotelInfo(
        searchParams: IHotelInfoRequestParams,
    ): Promise<IHotelInfo> {
        const attributionParams =
            this.hotelsMetaParamsBuilder.getAttributionParams();
        const debugPortalHostParams =
            this.hotelsMetaParamsBuilder.getDebugPortalHost();

        return this.hotelSearchAPI.getHotelInfo({
            ...debugPortalHostParams,
            ...attributionParams,
            ...searchParams,
        });
    }

    async getCalendarPrices(
        searchParams: ICalendarPricesRequest,
    ): Promise<ICalendarPricesResponse> {
        return this.hotelSearchAPI.getCalendarPrices(searchParams);
    }

    async getHotelImages(
        imagesParams: IHotelImagesRequestParams,
    ): Promise<IHotelImages> {
        return this.hotelSearchAPI.getHotelImages(imagesParams);
    }

    async getHotelReviews(
        reviewsParams: IHotelReviewsInfoRequestParams,
    ): Promise<IHotelReviewsInfo> {
        return this.hotelSearchAPI
            .getHotelReviews(reviewsParams)
            .then(prepareHotelReviews);
    }

    async addHotelReview(
        request: IAddHotelReviewRequest,
    ): Promise<IAddHotelReviewResponse> {
        return this.hotelSearchAPI.addHotelReview(request);
    }

    async editHotelReview(
        request: IEditHotelReviewRequest,
    ): Promise<IEditHotelReviewResponse> {
        return this.hotelSearchAPI.editHotelReview(request);
    }

    async deleteHotelReview(
        request: IDeleteHotelReviewRequest,
    ): Promise<IDeleteHotelReviewResponse> {
        return this.hotelSearchAPI.deleteHotelReview(request);
    }

    async uploadHotelReviewImage(): Promise<IUploadHotelReviewImageResponse> {
        // на сервере картинки только проксируютсся до бекенда методом proxyUploadHotelReviewImage
        throw new Error('Not implemented (only for browser)');
    }

    async proxyUploadHotelReviewImage(
        req: Request,
    ): Promise<IUploadHotelReviewImageResponse> {
        return this.hotelSearchAPI.proxyUploadHotelReviewImage(req);
    }

    async deleteHotelReviewImages(
        params: IDeleteHotelReviewImagesRequestQuery,
        request: IDeleteHotelReviewImagesRequestBody,
    ): Promise<IDeleteHotelReviewImagesResponse> {
        return this.hotelSearchAPI.deleteHotelReviewImages(params, request);
    }

    async getHotelOffers(
        offersParams: IHotelOffersInfoRequestParams,
    ): Promise<IHotelOffersInfo> {
        const attributionParams =
            this.hotelsMetaParamsBuilder.getAttributionParams();
        const debugPortalHostParams =
            this.hotelsMetaParamsBuilder.getDebugPortalHost();

        return this.hotelSearchAPI
            .getHotelOffers({
                ...debugPortalHostParams,
                ...attributionParams,
                ...offersParams,
            })
            .then(prepareHotelOffersInfo);
    }

    async getSimilarHotels(
        similarHotelsParams: ISimilarHotelsInfoRequestParams,
    ): Promise<ISimilarHotelsInfo> {
        const attributionParams =
            this.hotelsMetaParamsBuilder.getAttributionParams();

        return this.hotelSearchAPI
            .getSimilarHotels({...attributionParams, ...similarHotelsParams})
            .then(prepareSimilarHotelsInfo);
    }

    async setHotelReviewReaction(
        reaction: IHotelReviewReactionRequestParams,
    ): Promise<IHotelReviewReactionResponse> {
        return this.hotelSearchAPI.setHotelReviewReaction(reaction);
    }
}
