import {Readable} from 'stream';

import {
    IBookOffer,
    IFetchBookOfferParams,
} from 'server/api/HotelsBookAPI/types/IBookOffer';
import {
    IStartPaymentParams,
    TStartPayment,
} from 'server/api/HotelsBookAPI/types/TStartPayment';
import {IOrderParams, IOrder} from 'server/api/HotelsBookAPI/types/IOrder';
import {
    IOrderRefundableInfoParams,
    IOrderRefundableInfo,
} from 'server/api/HotelsBookAPI/types/IOrderRefundableInfo';
import {
    ICancelOrderParams,
    TCancelOrderType,
} from 'server/api/HotelsBookAPI/types/ICancelOrder';
import {IOrderPdfLinkResponse} from 'server/api/HotelsBookAPI/types/IOrderPdfLinkResponse';
import {
    ICheckVoucher,
    ICheckVoucherParams,
} from 'server/api/HotelsBookAPI/types/ICheckVoucher';
import {
    ESearchByOrderId,
    IDownloadVoucherParams,
} from 'server/api/HotelsBookAPI/types/IDownloadVoucher';
import {
    ICustomerCreateOrderRequest,
    ICreateOrder,
} from 'types/hotels/book/ICreateOrder';
import {
    IApplyPromoCodesRequest,
    IAppliedPromoCodesInfo,
} from 'types/hotels/book/IApplyPromoCodes';
import {IOrderStatusResponse} from 'server/api/HotelsBookAPI/types/IOrderResponse';
import {
    ITestBookOfferTokenRequestParams,
    ITestBookOfferTokenResponse,
} from 'server/api/HotelsBookAPI/types/ITestBookOfferToken';
import IGenerateBusinessTripPdfRequest from 'server/api/HotelsBookAPI/types/IGenerateBusinessTripPdfRequest';
import {ICheckBusinessTripPdfResponse} from 'server/api/HotelsBookAPI/types/ICheckBusinessTripPdfResponse';
import {ICheckBusinessTripPdfRequest} from 'server/api/HotelsBookAPI/types/ICheckBusinessTripPdfRequest';
import IDownloadBusinessTripPdfRequest from 'server/api/HotelsBookAPI/types/IDownloadBusinessTripPdfRequest';
import {IOrderPdfLinkRequest} from 'server/api/HotelsBookAPI/types/IOrderPdfLinkRequest';

import {ITokenHeader} from 'utilities/csrfToken/prepareTokenHeader';
import {prepareOrderResponse} from 'server/api/HotelsBookAPI/utilities/prepareOrder';
import {prepareOfferInfo} from 'server/api/HotelsBookAPI/utilities/prepareOfferInfo';
import {prepareCheckVoucher} from 'server/api/HotelsBookAPI/utilities/prepareCheckVoucher';
import {prepareCancelOrderResponse} from 'server/api/HotelsBookAPI/utilities/prepareCancelOrder';
import {prepareOrderRefundableInfoResponse} from 'server/api/HotelsBookAPI/utilities/prepareOrderRefundableInfo';
import {prepareStartPaymentResponse} from 'server/api/HotelsBookAPI/utilities/prepareStartPayment';
import {prepareCreateOrderResponse} from 'server/api/HotelsBookAPI/utilities/prepareCreateOrder';
import {prepareEstimateDiscountRequest} from './utilities/prepareEstimateDiscountRequest';
import prepareCheckBusinessTripPdf from 'server/api/HotelsBookAPI/utilities/prepareCheckBusinessTripPdf';
import {getDownloadVoucherPdfName} from 'server/api/HotelsBookAPI/utilities/prepareDownloadVoucher';
import getBusinessTripPdfName from 'server/api/HotelsBookAPI/utilities/getBusinessTripPdfName';

import {HotelsBookAPI} from 'server/api/HotelsBookAPI/HotelsBookAPI';
import {IDependencies} from 'server/getContainerConfig';
import {OrdersAPI} from 'server/api/OrdersAPI/OrdersAPI';
import {TestContextService} from 'server/services/TestContextService/TestContextService';

export interface IHotelsBookService {
    fetchOfferInfo(params: IFetchBookOfferParams): Promise<IBookOffer>;
    createOrder(
        order: ICustomerCreateOrderRequest,
        headers: ITokenHeader,
    ): Promise<ICreateOrder>;
    getOrder(params: IOrderParams): Promise<IOrder>;
    getOrderStatus(params: IOrderParams): Promise<IOrderStatusResponse>;
    estimateDiscount(
        params: IApplyPromoCodesRequest,
    ): Promise<IAppliedPromoCodesInfo>;
    checkVoucher(params: ICheckVoucherParams): Promise<ICheckVoucher>;
    removeOrder(id: string, headers: ITokenHeader): Promise<void>;
    startPayment(
        paymentParams: IStartPaymentParams,
        headers: ITokenHeader,
    ): Promise<TStartPayment>;
    getHotelsTestContextToken(
        params: ITestBookOfferTokenRequestParams,
    ): Promise<ITestBookOfferTokenResponse>;
}

export interface IDownloadPdf {
    pdf: Readable;
    fileName: string;
}

export class HotelsBookService implements IHotelsBookService {
    private hotelsBookAPI: HotelsBookAPI;
    private ordersAPI: OrdersAPI;
    private testContextService: TestContextService;

    constructor({hotelsBookAPI, ordersAPI, testContextService}: IDependencies) {
        this.hotelsBookAPI = hotelsBookAPI;
        this.ordersAPI = ordersAPI;
        this.testContextService = testContextService;
    }

    /* Fetch offer flow */

    async fetchOfferInfo(
        fetchOfferParams: IFetchBookOfferParams,
    ): Promise<IBookOffer> {
        return this.hotelsBookAPI
            .fetchOfferInfo(fetchOfferParams)
            .then(prepareOfferInfo);
    }

    /* Create order flow */

    async createOrder(
        createOrderParams: ICustomerCreateOrderRequest,
    ): Promise<ICreateOrder> {
        const paymentTestContextToken =
            await this.testContextService.getPaymentTestContextTokenIfNeeded(
                createOrderParams.paymentTestContextToken,
            );

        return this.hotelsBookAPI
            .createOrder({
                ...createOrderParams,
                paymentTestContextToken,
            })
            .then(prepareCreateOrderResponse);
    }

    async startPayment(
        startPaymentParams: IStartPaymentParams,
    ): Promise<TStartPayment> {
        const paymentTestContextToken =
            await this.testContextService.getStartPaymentTestContextTokenIfNeeded(
                startPaymentParams.paymentTestContextToken,
            );

        return this.hotelsBookAPI
            .startPayment({
                ...startPaymentParams,
                paymentTestContextToken,
            })
            .then(prepareStartPaymentResponse);
    }

    async getOrder(orderParams: IOrderParams): Promise<IOrder> {
        return this.hotelsBookAPI
            .getOrder(orderParams)
            .then(prepareOrderResponse);
    }

    async getOrderStatus(
        orderParams: IOrderParams,
    ): Promise<IOrderStatusResponse> {
        return this.hotelsBookAPI.getOrderStatus(orderParams);
    }

    async estimateDiscount(
        promoCodesInfo: IApplyPromoCodesRequest,
    ): Promise<IAppliedPromoCodesInfo> {
        const estimateDiscountParams =
            prepareEstimateDiscountRequest(promoCodesInfo);

        return this.hotelsBookAPI.estimateDiscount(estimateDiscountParams);
    }

    /* Cancel order flow */

    async getOrderRefundableInfo(
        orderParams: IOrderRefundableInfoParams,
    ): Promise<IOrderRefundableInfo | null> {
        return this.hotelsBookAPI
            .getOrderRefundableInfo(orderParams)
            .then(prepareOrderRefundableInfoResponse);
    }

    async cancelOrder(
        cancelOrderParams: ICancelOrderParams,
    ): Promise<TCancelOrderType> {
        return this.hotelsBookAPI
            .cancelOrder(cancelOrderParams)
            .then(prepareCancelOrderResponse);
    }

    /* Remove order */

    removeOrder = (id: string): Promise<void> =>
        this.hotelsBookAPI.removeOrder(id);

    /* Download business trip doc */

    async generateBusinessTripPdf(
        params: IGenerateBusinessTripPdfRequest,
    ): Promise<void> {
        return this.hotelsBookAPI.generateBusinessTripPdf(params);
    }

    async checkBusinessTripPdf(
        params: ICheckBusinessTripPdfRequest,
    ): Promise<ICheckBusinessTripPdfResponse> {
        const link = await this.getOrderPdfLink({orderId: params.id});

        return prepareCheckBusinessTripPdf(link);
    }

    async downloadBusinessTripPdf({
        id,
    }: IDownloadBusinessTripPdfRequest): Promise<IDownloadPdf> {
        const {id: orderId, yandexOrderId: prettyOrderId} =
            await this.ordersAPI.checkOrderAuthorization({id});
        const {businessTripDocUrl} = await this.getOrderPdfLink({orderId});

        if (!businessTripDocUrl) {
            throw new Error('Business trip doc is not ready');
        }

        return {
            fileName: getBusinessTripPdfName(prettyOrderId),
            pdf: await this.hotelsBookAPI.downloadPdf({
                url: businessTripDocUrl,
            }),
        };
    }

    /* Download voucher flow */

    async checkVoucher(
        voucherParams: ICheckVoucherParams,
    ): Promise<ICheckVoucher> {
        return this.getOrderPdfLink(voucherParams).then(prepareCheckVoucher);
    }

    async getOrderPdfLink(
        voucherParams: IOrderPdfLinkRequest,
    ): Promise<IOrderPdfLinkResponse> {
        return this.hotelsBookAPI.getOrderPdfLink(voucherParams);
    }

    async downloadVoucher(
        voucherParams: IDownloadVoucherParams,
    ): Promise<IDownloadPdf> {
        const {orderIdOrPrettyOrderId, searchBy} = voucherParams;
        const {id: orderId, yandexOrderId: prettyOrderId} =
            await this.ordersAPI.checkOrderAuthorization(
                searchBy === ESearchByOrderId.ORDER_ID
                    ? {id: orderIdOrPrettyOrderId}
                    : {yandex_order_id: orderIdOrPrettyOrderId},
            );
        const {url} = await this.getOrderPdfLink({orderId});

        return {
            fileName: getDownloadVoucherPdfName(prettyOrderId),
            pdf: await this.hotelsBookAPI.downloadPdf({url}),
        };
    }

    getHotelsTestContextToken(
        params: ITestBookOfferTokenRequestParams,
    ): Promise<ITestBookOfferTokenResponse> {
        return this.hotelsBookAPI.getTestBookOfferToken(params);
    }
}
