import {
    ISearchHotelsInfoRequestParams,
    ISearchHotelsInfoResponse,
} from 'server/api/HotelsSearchAPI/types/ISearchHotels';
import {
    ISearchSuggestRequestParams,
    ISearchSuggestResponse,
} from 'server/api/HotelsSearchAPI/types/ISearchSuggest';
import {
    ILogSearchSuggestRequestParams,
    ILogSearchSuggestResponse,
} from 'server/api/HotelsSearchAPI/types/ILogSearchSuggest';
import {
    ICountHotelsResponse,
    ICountHotelsRequestParams,
} from 'server/api/HotelsSearchAPI/types/ICountHotels';
import {
    IGeoRegionInfoRequestParams,
    IGeoRegionInfoResponse,
} from 'server/api/HotelsSearchAPI/types/IGeoRegionInfo';

import {prepareSearchSuggest} from 'server/api/HotelsSearchAPI/utilities/prepareSearchSuggest';
import {HotelsMetaParamsBuilder} from 'server/utilities/hotels/HotelsMetaParamsBuilder/HotelsMetaParamsBuilder';

import {HotelsSearchAPI} from 'server/api/HotelsSearchAPI/HotelsSearchAPI';
import {IDependencies} from 'server/getContainerConfig';

type TSearchHotels = (
    searchParams: ISearchHotelsInfoRequestParams,
) => Promise<ISearchHotelsInfoResponse>;

type TSearchSuggest = (
    suggestParams: ISearchSuggestRequestParams,
) => Promise<ISearchSuggestResponse>;

export type TLogSearchSuggest = (
    logSuggestParams: ILogSearchSuggestRequestParams,
) => Promise<ILogSearchSuggestResponse>;

type TCountHotels = (
    countParams: ICountHotelsRequestParams,
) => Promise<ICountHotelsResponse>;

export type TGeoRegionType = (
    regionParams: IGeoRegionInfoRequestParams,
) => Promise<IGeoRegionInfoResponse>;

export interface IHotelsSearchService {
    searchHotels: TSearchHotels;
    searchSuggest: TSearchSuggest;
    logSearchSuggest: TLogSearchSuggest;
    countHotels: TCountHotels;
    getGeoRegionInfo: TGeoRegionType;
}

export class HotelsSearchService implements IHotelsSearchService {
    private hotelsSearchAPI: HotelsSearchAPI;
    private hotelsMetaParamsBuilder: HotelsMetaParamsBuilder;

    constructor({hotelsSearchAPI, hotelsMetaParamsBuilder}: IDependencies) {
        this.hotelsSearchAPI = hotelsSearchAPI;
        this.hotelsMetaParamsBuilder = hotelsMetaParamsBuilder;
    }

    async searchSuggest(
        suggestParams: ISearchSuggestRequestParams,
    ): Promise<ISearchSuggestResponse> {
        return this.hotelsSearchAPI
            .searchSuggest(suggestParams)
            .then(prepareSearchSuggest);
    }

    async logSearchSuggest(
        logSuggestParams: ILogSearchSuggestRequestParams,
    ): Promise<ILogSearchSuggestResponse> {
        return this.hotelsSearchAPI.logSearchSuggest(logSuggestParams);
    }

    async searchHotels(
        searchPollingParams: ISearchHotelsInfoRequestParams,
    ): Promise<ISearchHotelsInfoResponse> {
        const debugPortalHost =
            this.hotelsMetaParamsBuilder.getDebugPortalHost();
        const attributionParams =
            this.hotelsMetaParamsBuilder.getAttributionParams();

        return this.hotelsSearchAPI.searchHotels({
            ...debugPortalHost,
            ...attributionParams,
            ...searchPollingParams,
        });
    }

    async countHotels(
        countParams: ICountHotelsRequestParams,
    ): Promise<ICountHotelsResponse> {
        return this.hotelsSearchAPI.countHotels(countParams);
    }

    async getGeoRegionInfo(
        regionParams: IGeoRegionInfoRequestParams,
    ): Promise<IGeoRegionInfoResponse> {
        return this.hotelsSearchAPI.getGeoRegionInfo(regionParams);
    }
}
