import {INotebookGetPassengersService} from 'server/services/NotebookService/types/INotebookGetPassengersService';
import {INotebookGetTravelerService} from 'server/services/NotebookService/types/INotebookGetTravelerService';
import {INotebookGetPassengerService} from 'server/services/NotebookService/types/INotebookGetPassengerService';
import {INotebookGetPassengerDocumentsService} from 'server/services/NotebookService/types/INotebookGetPassengerDocumentsService';
import {INotebookGetPassengerDocumentService} from 'server/services/NotebookService/types/INotebookGetPassengerDocumentService';
import {INotebookGetPassengerBonusCardsService} from 'server/services/NotebookService/types/INotebookGetPassengerBonusCardsService';
import {INotebookGetPassengerBonusCardService} from 'server/services/NotebookService/types/INotebookGetPassengerBonusCardService';
import {INotebookGetDocumentTypesService} from 'server/services/NotebookService/types/INotebookGetDocumentTypesService';
import {INotebookCreateOrUpdateTravelerService} from 'server/services/NotebookService/types/INotebookCreateOrUpdateTravelerService';
import {INotebookCreatePassengerService} from 'server/services/NotebookService/types/INotebookCreatePassengerService';
import {INotebookEditPassengerService} from 'server/services/NotebookService/types/INotebookEditPassengerService';
import {INotebookDeletePassengerService} from 'server/services/NotebookService/types/INotebookDeletePassengerService';
import {INotebookEditPassengerDocumentService} from 'server/services/NotebookService/types/INotebookEditPassengerDocumentService';
import {INotebookCreatePassengerDocumentService} from 'server/services/NotebookService/types/INotebookCreatePassengerDocumentService';
import {INotebookDeletePassengerDocumentService} from 'server/services/NotebookService/types/INotebookDeletePassengerDocumentService';
import {INotebookCreatePassengerBonusCardService} from 'server/services/NotebookService/types/INotebookCreatePassengerBonusCardService';
import {INotebookEditPassengerBonusCardService} from 'server/services/NotebookService/types/INotebookEditPassengerBonusCardService';
import {INotebookDeletePassengerBonusCardService} from 'server/services/NotebookService/types/INotebookDeletePassengerBonusCardService';
import {INotebookSaveTravelerAndPassengersService} from 'server/services/NotebookService/types/INotebookSaveTravelerAndPassengersService';
import {ENotebookGetPassengerFields} from 'server/api/NotebookApi/types/getPassenger';

import getBadRequestError from 'server/services/NotebookService/utilities/getBadRequestError';
import findOrCreatePassenger from 'server/services/NotebookService/utilities/findOrCreatePassenger';
import findOrCreatePassengerDocument from 'server/services/NotebookService/utilities/findOrCreatePassengerDocument';
import findOrCreatePassengerBonusCard from 'server/services/NotebookService/utilities/findOrCreatePassengerBonusCard';

import {IDependencies} from 'server/getContainerConfig';
import {NotebookApi} from 'server/api/NotebookApi/NotebookApi';

export interface INotebookService {
    getTraveler: INotebookGetTravelerService;
    createOrUpdateTraveler: INotebookCreateOrUpdateTravelerService;

    getPassengers: INotebookGetPassengersService;
    getPassenger: INotebookGetPassengerService;
    createPassenger: INotebookCreatePassengerService;
    editPassenger: INotebookEditPassengerService;
    deletePassenger: INotebookDeletePassengerService;

    getPassengerDocuments: INotebookGetPassengerDocumentsService;
    getPassengerDocument: INotebookGetPassengerDocumentService;
    createPassengerDocument: INotebookCreatePassengerDocumentService;
    editPassengerDocument: INotebookEditPassengerDocumentService;
    deletePassengerDocument: INotebookDeletePassengerDocumentService;

    getPassengerBonusCards: INotebookGetPassengerBonusCardsService;
    getPassengerBonusCard: INotebookGetPassengerBonusCardService;
    createPassengerBonusCard: INotebookCreatePassengerBonusCardService;
    editPassengerBonusCard: INotebookEditPassengerBonusCardService;
    deletePassengerBonusCard: INotebookDeletePassengerBonusCardService;

    getDocumentTypes: INotebookGetDocumentTypesService;

    saveTravelerAndPassengers: INotebookSaveTravelerAndPassengersService;
}

function checkUid(uid: string | undefined): string {
    if (!uid) {
        throw getBadRequestError('No uid');
    }

    return uid;
}

export class NotebookService implements INotebookService {
    private readonly notebookApi: NotebookApi;
    private readonly uid?: string;

    constructor({notebookApi, userInfo}: IDependencies) {
        this.notebookApi = notebookApi;

        this.uid = userInfo && 'uid' in userInfo ? userInfo.uid : undefined;
    }

    getTraveler: INotebookGetTravelerService = async params => {
        const uid = checkUid(this.uid);

        return this.notebookApi.getTraveler({uid, ...params});
    };

    createOrUpdateTraveler: INotebookCreateOrUpdateTravelerService =
        async params => {
            const uid = checkUid(this.uid);

            return this.notebookApi.createOrUpdateTraveler({uid, ...params});
        };

    getPassengers: INotebookGetPassengersService = async params => {
        const uid = checkUid(this.uid);

        return this.notebookApi.getPassengers({uid, ...params});
    };

    getPassenger: INotebookGetPassengerService = async params => {
        const uid = checkUid(this.uid);

        return this.notebookApi.getPassenger({uid, ...params});
    };

    createPassenger: INotebookCreatePassengerService = async params => {
        const uid = checkUid(this.uid);

        return this.notebookApi.createPassenger({uid, ...params});
    };

    editPassenger: INotebookEditPassengerService = async params => {
        const uid = checkUid(this.uid);

        return this.notebookApi.editPassenger({uid, ...params});
    };

    deletePassenger: INotebookDeletePassengerService = async params => {
        const uid = checkUid(this.uid);

        return this.notebookApi.deletePassenger({uid, ...params});
    };

    getPassengerDocuments: INotebookGetPassengerDocumentsService =
        async params => {
            const uid = checkUid(this.uid);

            return this.notebookApi.getPassengerDocuments({uid, ...params});
        };

    getPassengerDocument: INotebookGetPassengerDocumentService =
        async params => {
            const uid = checkUid(this.uid);

            return this.notebookApi.getPassengerDocument({uid, ...params});
        };

    createPassengerDocument: INotebookCreatePassengerDocumentService =
        async params => {
            const uid = checkUid(this.uid);

            return this.notebookApi.createPassengerDocument({uid, ...params});
        };

    editPassengerDocument: INotebookEditPassengerDocumentService =
        async params => {
            const uid = checkUid(this.uid);

            return this.notebookApi.editPassengerDocument({uid, ...params});
        };

    deletePassengerDocument: INotebookDeletePassengerDocumentService =
        async params => {
            const uid = checkUid(this.uid);

            return this.notebookApi.deletePassengerDocument({uid, ...params});
        };

    getPassengerBonusCards: INotebookGetPassengerBonusCardsService =
        async params => {
            const uid = checkUid(this.uid);

            return this.notebookApi.getPassengerBonusCards({uid, ...params});
        };

    getPassengerBonusCard: INotebookGetPassengerBonusCardService =
        async params => {
            const uid = checkUid(this.uid);

            return this.notebookApi.getPassengerBonusCard({uid, ...params});
        };

    createPassengerBonusCard: INotebookCreatePassengerBonusCardService =
        async params => {
            const uid = checkUid(this.uid);

            return this.notebookApi.createPassengerBonusCard({uid, ...params});
        };

    editPassengerBonusCard: INotebookEditPassengerBonusCardService =
        async params => {
            const uid = checkUid(this.uid);

            return this.notebookApi.editPassengerBonusCard({uid, ...params});
        };

    deletePassengerBonusCard: INotebookDeletePassengerBonusCardService =
        async params => {
            const uid = checkUid(this.uid);

            return this.notebookApi.deletePassengerBonusCard({uid, ...params});
        };

    getDocumentTypes: INotebookGetDocumentTypesService = async () => {
        return this.notebookApi.getDocumentTypes();
    };

    saveTravelerAndPassengers: INotebookSaveTravelerAndPassengersService =
        async params => {
            const {traveler, passengers} = params;

            await this.createOrUpdateTraveler({traveler});

            const existedPassengers = await this.getPassengers({
                fields: ENotebookGetPassengerFields.DOCUMENTS_AND_BONUS_CARDS,
            });

            const passengerMutationPromises = passengers.map(
                async (passenger): Promise<void> => {
                    const existedPassenger = await findOrCreatePassenger({
                        existedPassengers,
                        passenger,
                        createPassenger: this.createPassenger,
                    });

                    await findOrCreatePassengerDocument({
                        existedPassenger,
                        document: passenger.document,
                        createDocument: this.createPassengerDocument,
                    });

                    await Promise.all(
                        passenger.bonusCards?.map(bonusCard =>
                            findOrCreatePassengerBonusCard({
                                existedPassenger,
                                bonusCard,
                                createBonusCard: this.createPassengerBonusCard,
                            }),
                        ) || [],
                    );
                },
            );

            await Promise.all(passengerMutationPromises);
        };
}
