import {
    INotifierPromoConfig,
    INotifierSubscribeApiParams,
} from 'server/api/NotifierApi/types';
import {ESubscriptionAuthType} from 'types/subscription/ESubscriptionAuthType';
import {ESubscriptionVerticalName} from 'types/subscription/ESubscriptionVerticalName';

import {isUnknownAxiosError} from 'utilities/error';

import {IDependencies} from 'server/getContainerConfig';
import {NotifierApi} from 'server/api/NotifierApi/NotifierApi';

export interface INotifierService {
    getStatus: (email: string) => Promise<boolean | null>;
    getPromoConfig: (
        vertical: ESubscriptionVerticalName,
        email?: string,
    ) => Promise<INotifierPromoConfig>;
    subscribe: (
        params: Omit<INotifierSubscribeApiParams, 'authType' | 'authValue'>,
    ) => Promise<void>;
    unsubscribe: (id: string) => Promise<void>;
}

export class NotifierService implements INotifierService {
    private notifierApi: NotifierApi;
    private readonly authType: ESubscriptionAuthType;
    private readonly authValue: string;

    constructor({notifierApi, userInfo, yandexuid}: IDependencies) {
        const passportUid =
            userInfo && 'uid' in userInfo ? userInfo.uid : undefined;

        this.notifierApi = notifierApi;
        this.authType = passportUid
            ? ESubscriptionAuthType.passportId
            : ESubscriptionAuthType.yandexUid;
        this.authValue = passportUid ?? yandexuid;
    }

    async getPromoConfig(
        vertical: ESubscriptionVerticalName,
        email?: string,
    ): Promise<INotifierPromoConfig> {
        return this.notifierApi.getPromoConfig({
            vertical,
            email,
            authType: this.authType,
            authValue: this.authValue,
        });
    }

    async getStatus(email: string): Promise<boolean | null> {
        try {
            const status = await this.notifierApi.getStatus(
                email,
                this.authType,
                this.authValue,
            );

            return Boolean(status && status.isSubscribed);
        } catch (err) {
            return isUnknownAxiosError(err) && err.response?.status === 404
                ? false
                : null;
        }
    }

    async subscribe(
        params: Omit<INotifierSubscribeApiParams, 'authType' | 'authValue'>,
    ): Promise<void> {
        await this.notifierApi.subscribe({
            ...params,
            authType: this.authType,
            authValue: this.authValue,
        });
    }

    async unsubscribe(id: string): Promise<void> {
        await this.notifierApi.unsubscribe(id, this.authType, this.authValue);
    }
}
