import {LANG} from 'constants/langCodes';

import {IRaspParseContextApiParams} from 'server/api/RaspApi/types/IRaspParseContextApiParams';
import {IRaspParseContextService} from 'server/services/RaspService/types/IRaspParseContextService';
import {IRaspGetTrainOrderSegmentService} from 'server/services/RaspService/types/IRaspGetTrainOrderSegmentService';
import {IRaspGetCountriesService} from './types/IRaspGetCountriesService';

import {canonicalizeContext} from 'projects/trains/lib/context/canonicalizeContext';

import {RaspApi} from 'server/api/RaspApi/RaspApi';
import {RaspCacheApi} from 'server/api/RaspCacheApi/RaspCacheApi';
import {IDependencies} from 'server/getContainerConfig';
import prepareRaspGetTrainOrderSegmentApiParams from 'server/services/RaspService/data-transfromers/prepareRaspGetTrainOrderSegmentApiParams';
import {TrainsSearchService} from 'server/services/TrainsSearchService/TrainsSearchService';

export interface IRaspService {
    parseContext: IRaspParseContextService;
    getTrainOrderSegment: IRaspGetTrainOrderSegmentService;
}

export class RaspService implements IRaspService {
    private raspApi: RaspApi;
    private raspCacheApi: RaspCacheApi;
    private trainsSearchService: TrainsSearchService;

    constructor({raspApi, raspCacheApi, trainsSearchService}: IDependencies) {
        this.raspApi = raspApi;
        this.raspCacheApi = raspCacheApi;
        this.trainsSearchService = trainsSearchService;
    }

    parseContext: IRaspParseContextService = async ({
        language = LANG.RU,
        transport,
        ...params
    }) => {
        const preparedParams: IRaspParseContextApiParams = {
            ...params,
            national_version: language,
            t_type: transport,
            language,
        };

        const [parsedContext, canonical] = await Promise.all([
            this.raspCacheApi.parseContext(preparedParams),
            this.trainsSearchService.getCanonical({
                transport,
                fromSlug: params.from_slug,
                toSlug: params.to_slug,
            }),
        ]);

        return canonicalizeContext(parsedContext, canonical);
    };

    getTrainOrderSegment: IRaspGetTrainOrderSegmentService = async params => {
        const {result} = await this.raspApi.getTrainOrderSegment(
            prepareRaspGetTrainOrderSegmentApiParams(params),
        );

        return result;
    };

    getCountries: IRaspGetCountriesService = async () => {
        const countries = await this.raspApi.getCountries();

        return {
            countries: countries.map(country => ({
                ...country,
                value: country.geoId,
                data: country.title,
            })),
        };
    };
}
