import NodeCache from 'node-cache';

import {ENodeCacheKey} from 'server/constants/eNodeCacheKey';

import {ISimpleTextReview} from 'types/common/ISimpleTextReview';
import {IFormattedTrainsReviews} from 'server/services/ReviewsService/types/IFormattedTrainsReviews';
import {IReviewsGetTrainsReviewsApiParams} from 'server/api/ReviewsApi/types/IReviewsGetTrainsReviewsApiParams';

import {getRecentReviews} from 'server/services/ReviewsService/utilities/getRecentReviews';
import {formatTrainReviews} from 'server/services/ReviewsService/utilities/formatTrainReviews';

import {IDependencies} from 'server/getContainerConfig';
import {ReviewsApi} from 'server/api/ReviewsApi/ReviewsApi';

const cache = new NodeCache({stdTTL: 60 * 60, useClones: false});

export interface IReviewsService {
    getTrainsReviews: (
        params: IReviewsGetTrainsReviewsApiParams,
    ) => Promise<ISimpleTextReview[]>;
}

export class ReviewsService implements IReviewsService {
    private reviewsApi: ReviewsApi;

    constructor({reviewsApi}: IDependencies) {
        this.reviewsApi = reviewsApi;
    }

    async getTrainsReviews({
        fromSlug,
        toSlug,
    }: IReviewsGetTrainsReviewsApiParams): Promise<ISimpleTextReview[]> {
        const direction = `${fromSlug}--${toSlug}`;
        let reviewsData = cache.get<IFormattedTrainsReviews>(
            ENodeCacheKey.TRAINS_REVIEWS,
        );

        if (!reviewsData) {
            reviewsData = formatTrainReviews(
                await this.reviewsApi.getTrainsReviews(),
            );

            cache.set<IFormattedTrainsReviews>(
                ENodeCacheKey.TRAINS_REVIEWS,
                reviewsData,
            );
        }

        return getRecentReviews(reviewsData, direction);
    }
}
