import NodeCache from 'node-cache';

import {ENodeCacheKey} from 'server/constants/eNodeCacheKey';

import {EContentPageType} from 'projects/journal/types/EContentPageType';
import {EContentPageFormat} from 'projects/journal/types/EContentPageFormat';
import {
    IMetaArticle,
    IGetContextPageTag,
    IGetContextPageArticle,
} from 'server/api/TemplatorApi/types/IGetContextPageResponse';
import {EGetContextPageZoom} from 'server/api/TemplatorApi/types/IGetContextPageRequestParams';
import {IGetArticleRequestParams} from 'server/services/TemplatorService/types/IGetArticleRequestParams';
import {IGetRootMetaRequestParams} from 'server/services/TemplatorService/types/IGetRootMetaRequestParams';
import {IGetArticlesRequestParams} from 'server/services/TemplatorService/types/IGetArticlesRequestParams';
import {IGetTagsListRequestParams} from 'server/services/TemplatorService/types/IGetTagsListRequestParams';
import {IGetTagRequestParams} from 'server/services/TemplatorService/types/IGetTagRequestParams';

import {TemplatorApi} from 'server/api/TemplatorApi/TemplatorApi';

import {IDependencies} from '../../getContainerConfig';

export type TGetRootMetaService = (
    params: IGetRootMetaRequestParams,
) => Promise<IMetaArticle>;
export type TGetArticleService = (
    params: IGetArticleRequestParams,
) => Promise<IGetContextPageArticle | null>;
export type TGetArticlesService = (
    params: IGetArticlesRequestParams,
) => Promise<IGetContextPageArticle[]>;
export type TGetTagsListService = (
    params: IGetTagsListRequestParams,
) => Promise<IGetContextPageTag[]>;
export type TGetTagService = (
    params: IGetTagRequestParams,
) => Promise<IGetContextPageTag>;

export interface ITemplatorService {
    getRootMeta: TGetRootMetaService;
    getArticle: TGetArticleService;
    getArticles: TGetArticlesService;
    getTagsList: TGetTagsListService;
    getTagInfo: TGetTagService;
}

const cache = new NodeCache({stdTTL: 5 * 60});

export class TemplatorService implements ITemplatorService {
    private templatorApi: TemplatorApi;

    constructor({templatorApi}: IDependencies) {
        this.templatorApi = templatorApi;
    }

    async getRootMeta(
        params: IGetRootMetaRequestParams,
    ): Promise<IMetaArticle> {
        return (
            await this.templatorApi.getContextPage<IMetaArticle>({
                device: params.device,
                ds: params.ds,
                semantic_id: params.semantic_id,
                templator_content: params.templator_content,
                format: EContentPageFormat.JSON,
                type: EContentPageType.JOURNAL,
                zoom: EGetContextPageZoom.FULL,
            })
        ).result[0];
    }

    async getArticle(
        params: IGetArticleRequestParams,
    ): Promise<IGetContextPageArticle> {
        return (
            await this.templatorApi.getContextPage<IGetContextPageArticle>({
                semantic_id: params.semantic_id,
                ds: params.ds,
                templator_content: params.templator_content,
                device: params.device,
                format: EContentPageFormat.JSON,
                type: EContentPageType.ARTICLE,
                zoom: EGetContextPageZoom.FULL,
            })
        ).result[0];
    }

    async getArticles(
        params: IGetArticlesRequestParams,
    ): Promise<IGetContextPageArticle[]> {
        const cacheKey = [
            ENodeCacheKey.CONTENT_PAGE,
            params.tag,
            params.ds,
            params.templator_content,
            params.device,
        ]
            .filter(Boolean)
            .join('-');
        const dataFromCache = cache.get<IGetContextPageArticle[] | undefined>(
            cacheKey,
        );

        if (dataFromCache) {
            return dataFromCache;
        }

        const {result} =
            await this.templatorApi.getContextPage<IGetContextPageArticle>({
                device: params.device,
                ds: params.ds,
                templator_content: params.templator_content,
                format: EContentPageFormat.JSON,
                type: EContentPageType.ARTICLE,
                zoom: EGetContextPageZoom.ENTRYPOINTS,
                tag: params.tag || undefined,
            });

        cache.set(cacheKey, result);

        return result;
    }

    async getTagsList(
        params: IGetTagsListRequestParams,
    ): Promise<IGetContextPageTag[]> {
        const {result} =
            await this.templatorApi.getContextPage<IGetContextPageTag>({
                device: params.device,
                ds: params.ds,
                templator_content: params.templator_content,
                format: EContentPageFormat.JSON,
                type: EContentPageType.TAGS,
                zoom: EGetContextPageZoom.ENTRYPOINTS,
            });

        return result;
    }

    async getTagInfo(
        params: IGetTagRequestParams,
    ): Promise<IGetContextPageTag> {
        const {result} =
            await this.templatorApi.getContextPage<IGetContextPageTag>({
                device: params.device,
                ds: params.ds,
                templator_content: params.templator_content,
                format: EContentPageFormat.JSON,
                type: EContentPageType.TAGS,
                zoom: EGetContextPageZoom.FULL,
                semantic_id: params.tagSlug,
            });

        return result[0];
    }
}
