import {LOYALTY_CARDS_ORDER_TO_TRAVELER_MATCH} from 'projects/trains/lib/order/traveler/constants';

import {ISaveDocumentToTravelerNotebookPassenger} from 'server/services/TicketsTravelersNotebookService/types/common';
import {ITrainsLoyaltyCard} from 'server/api/TrainsApi/types/ITrainsLoyaltyCard';
import {IBonusCardDTO} from 'server/api/TravelersApi/types/IBonusCardDTO';
import {TBlackboxUserInfo} from 'types/common/userInfo/TBlackboxUserInfo';
import {ISaveDocumentsAndTravelerToNotebookApiParams} from 'server/services/TicketsTravelersNotebookService/types/ISaveDocumentsAndTravelerToNotebookApiParams';

import {ILogger} from 'server/utilities/Logger';
import prepareDocumentToStore from './utilities/prepareDocumentToStore';
import getExistedPassenger from './utilities/getExistedPassenger';
import passengerHasDocument from './utilities/passengerHasDocument';
import getTravelerPassengerLoyaltyCardsToCreateAndDelete from './utilities/getTravelerPassengerLoyaltyCardsToCreateAndDelete';
import getDataToCreateTravelerDocument from './utilities/getDataToCreateTravelerDocument';
import {isPatronymicEmpty} from 'projects/trains/lib/order/fields/patronymic';
import getLoyaltyCardName from 'projects/trains/lib/getLoyaltyCardName';
import {unknownToErrorOrUndefined} from 'utilities/error';

import {TravelersApi} from 'server/api/TravelersApi/TravelersApi';
import {AccountsService} from 'server/services/AccountsService/AccountsService';
import {IDependencies} from 'server/getContainerConfig';

interface ITravelPassengerCreationInfo {
    id: string | null;
    needToCreateDocument: boolean;
    loyaltyCardsToCreate: (ITrainsLoyaltyCard | null)[];
    loyaltyCardsToDelete: IBonusCardDTO[];
}

/**
 * TODO: тут содержится дублирующаяся с авиа-бронированием логика
 * по сохранению документов в записную книжку.
 *
 * Этот класс создаётся на время перехода, и в будущем должен быть заменён
 * общим для портала путешествия сервисом для сохранения документов в
 * записную книжку по документам из заказка
 */
export class TicketsTravelersNotebookService {
    private logger: ILogger;

    private accountsService: AccountsService;

    private travelersApi: TravelersApi;

    private userInfo: TBlackboxUserInfo;

    private userUid?: string;

    constructor({
        accountsService,
        travelersApi,
        userInfo,
        logger,
    }: IDependencies) {
        this.logger = logger;
        this.accountsService = accountsService;
        this.travelersApi = travelersApi;
        this.userInfo = userInfo;
        this.userUid = userInfo && 'uid' in userInfo ? userInfo.uid : undefined;
    }

    /**
     * @deprecated - использовать notebookService
     */
    async saveDocumentsToNotebook(
        passengers: ISaveDocumentToTravelerNotebookPassenger[],
    ): Promise<void> {
        const userInfo = this.userInfo;

        if (!userInfo || !('uid' in userInfo)) {
            return;
        }

        const preparedOrderPassengers = passengers.map(passenger => ({
            title: `${passenger.firstName}${
                isPatronymicEmpty(passenger.patronymic)
                    ? ''
                    : ` ${passenger.patronymic}`
            } ${passenger.lastName}`,
            gender: passenger.gender,
            birthDate: passenger.birthDate,
            document: prepareDocumentToStore(passenger),
            loyaltyCards: passenger.loyaltyCards,
            availableLoyaltyCards: passenger.availableLoyaltyCards,
        }));

        const travelerPassengersWithDocuments =
            await this.accountsService.getPassengersWithDocumentsAndBonusCards();

        // Создаем пассажиров, если их до этого не было. Собираем информацию, нужно ли создавать документы и бонусные карты.
        const passengersCreationInfo = await Promise.all(
            preparedOrderPassengers.map(
                async (
                    preparedOrderPassenger,
                ): Promise<ITravelPassengerCreationInfo> => {
                    // Ищем пассажира по существующим документам
                    const existedPassenger = getExistedPassenger(
                        travelerPassengersWithDocuments,
                        preparedOrderPassenger,
                    );

                    // Если существует пассажир с документами, которые соответсвуют его имени
                    if (existedPassenger) {
                        const passengerDataToCreate: ITravelPassengerCreationInfo =
                            {
                                id: null,
                                needToCreateDocument: false,
                                loyaltyCardsToCreate: [],
                                loyaltyCardsToDelete: [],
                            };

                        // Добавлен ли уже документ
                        const hasDocument = passengerHasDocument(
                            existedPassenger,
                            preparedOrderPassenger.document,
                        );

                        const {loyaltyCardsToCreate, loyaltyCardsToDelete} =
                            getTravelerPassengerLoyaltyCardsToCreateAndDelete(
                                preparedOrderPassenger.availableLoyaltyCards,
                                preparedOrderPassenger.loyaltyCards,
                                existedPassenger.bonus_cards,
                            );

                        // Если документ еще не был добавлен к пассажиру
                        if (!hasDocument) {
                            passengerDataToCreate.needToCreateDocument = true;
                        }

                        return {
                            ...passengerDataToCreate,
                            id: existedPassenger.id,
                            loyaltyCardsToCreate,
                            loyaltyCardsToDelete,
                        };
                    }

                    // Если пассажира с документами, которые соответсуют его имени, не найдено, то создаем его
                    const newPassenger =
                        await this.travelersApi.createPassenger(userInfo.uid, {
                            title: preparedOrderPassenger.title,
                            gender: preparedOrderPassenger.gender as any,
                            birth_date: preparedOrderPassenger.birthDate,
                        });

                    return {
                        id: newPassenger.id,
                        needToCreateDocument: true,
                        loyaltyCardsToCreate:
                            preparedOrderPassenger.loyaltyCards,
                        loyaltyCardsToDelete: [],
                    };
                },
            ),
        );

        passengersCreationInfo.forEach(
            (
                {
                    id: passengerId,
                    needToCreateDocument,
                    loyaltyCardsToCreate,
                    loyaltyCardsToDelete,
                },
                index,
            ) => {
                if (!passengerId) {
                    return;
                }

                // Добавляем документ и лояльные карты к пассажиру
                const preparedOrderPassenger = preparedOrderPassengers[index];

                if (needToCreateDocument) {
                    this.travelersApi.createDocument(
                        userInfo.uid,
                        passengerId,
                        getDataToCreateTravelerDocument(
                            preparedOrderPassenger.document,
                        ),
                    );
                }

                loyaltyCardsToCreate.forEach(loyaltyCard => {
                    if (!loyaltyCard) {
                        return;
                    }

                    this.travelersApi.createBonusCard(
                        userInfo.uid,
                        passengerId,
                        {
                            title: getLoyaltyCardName(loyaltyCard.type),
                            type: LOYALTY_CARDS_ORDER_TO_TRAVELER_MATCH[
                                loyaltyCard.type
                            ],
                            number: loyaltyCard.number,
                        },
                    );
                });

                loyaltyCardsToDelete.forEach(loyaltyCard => {
                    this.travelersApi.deleteBonusCard(
                        userInfo.uid,
                        passengerId,
                        loyaltyCard.id,
                    );
                });
            },
        );
    }

    /**
     * @deprecated - использовать notebookService
     */
    async saveDocumentsAndTravelerToNotebook({
        traveler,
        passengers,
    }: ISaveDocumentsAndTravelerToNotebookApiParams): Promise<void> {
        if (!this.userUid) {
            return;
        }

        try {
            await this.travelersApi.createOrUpdateTraveler(this.userUid, {
                ...traveler,
                agree: true,
            });

            await this.saveDocumentsToNotebook(passengers);
        } catch (err) {
            this.logger.logError(
                'Trains: saveDocumentsAndTravelerToNotebook',
                unknownToErrorOrUndefined(err),
            );
        }
    }
}
