import {ETransport} from 'types/common/ETransport';
import {
    ITrainsSearchInfoResponse,
    ITrainsSearchRequestParams,
} from 'server/api/TrainsSearchApi/types/ITrainsSearch';
import {
    ITrainsCrossLinksRequestParams,
    ITrainsCrossLinksResponse,
} from 'server/api/TrainsSearchApi/types/ITrainsCrossLinks';
import {ITrainsCanonicalResponse} from 'server/api/TrainsSearchApi/types/ITrainsCanonical';
import {
    ITrainsPriceCalendarRequestParams,
    ITrainsPriceCalendarResponse,
} from 'server/api/TrainsSearchApi/types/ITrainsPriceCalendar';

import {IDependencies} from 'server/getContainerConfig';
import {TrainsSearchApi} from 'server/api/TrainsSearchApi/TrainsSearchApi';
import {TestContextService} from 'server/services/TestContextService/TestContextService';

type TTrainsSearch = (
    searchParams: ITrainsSearchRequestParams,
) => Promise<ITrainsSearchInfoResponse>;
type TTrainsCrossLinks = (
    params: ITrainsCrossLinksRequestParams,
) => Promise<ITrainsCrossLinksResponse>;
type TTrainsPriceCalendar = (
    params: ITrainsPriceCalendarRequestParams,
) => Promise<ITrainsPriceCalendarResponse>;

export interface ITrainsSearchService {
    search: TTrainsSearch;
    getCrossLinks: TTrainsCrossLinks;
    priceCalendar: TTrainsPriceCalendar;
}

export class TrainsSearchService implements ITrainsSearchService {
    private trainsSearchApi: TrainsSearchApi;
    private testContextService: TestContextService;

    constructor({trainsSearchApi, testContextService}: IDependencies) {
        this.trainsSearchApi = trainsSearchApi;
        this.testContextService = testContextService;
    }

    async search(
        searchParams: ITrainsSearchRequestParams,
    ): Promise<ITrainsSearchInfoResponse> {
        const mockImTestContextParams =
            await this.testContextService.getTrainsMockImTestContextParamsIfNeed();

        return this.trainsSearchApi.search({
            ...searchParams,
            ...mockImTestContextParams,
        });
    }

    async getCanonical({
        transport,
        fromSlug,
        toSlug,
    }: {
        transport: ETransport;
        fromSlug?: string;
        toSlug?: string;
    }): Promise<Nullable<ITrainsCanonicalResponse>> {
        if (transport !== ETransport.TRAIN || !fromSlug || !toSlug) {
            return null;
        }

        try {
            return await this.trainsSearchApi.getCanonical({
                fromSlug,
                toSlug,
            });
        } catch (err) {
            // Паденние этой ручки не критично и не должно влиять на пользователей
            return null;
        }
    }

    getCrossLinks(
        params: ITrainsCrossLinksRequestParams,
    ): Promise<ITrainsCrossLinksResponse> {
        return this.trainsSearchApi.getCrossLinks({
            fromKey: params.fromKey,
            toKey: params.toKey,
        });
    }

    priceCalendar(
        params: ITrainsPriceCalendarRequestParams,
    ): Promise<ITrainsPriceCalendarResponse> {
        return this.trainsSearchApi.priceCalendar(params);
    }
}
