import {Readable} from 'stream';

import {LANG} from 'constants/langCodes';
import {ETrainsPartner} from 'projects/trains/constants/trainPartners';

import {Request} from '@yandex-data-ui/core/lib/types';
import {IActivePartnersParams} from './types/IActivePartnersParams';
import {ITrainsLogBanditApiParams} from 'server/api/TrainsApi/types/ITrainsLogBanditApiParams';
import {IUserParams} from 'server/services/TrainsService/types/IUserParams';
import {TBlackboxUserInfo} from 'types/common/userInfo/TBlackboxUserInfo';
import {
    ITrainsSaveOfferServiceParams,
    ITrainsSaveOfferServiceResponse,
} from 'server/services/TrainsService/types/ITrainsSaveOfferService';
import {ITrainsGetSuggestsService} from 'server/services/TrainsService/types/ITrainsGetSuggestsService';
import {ITrainsSaveDocumentsAndTravelerToNotebookServiceParams} from 'server/services/TrainsService/types/ITrainsSaveDocumentsAndTravelerToNotebookServiceParams';
import {ITrainsAddInsuranceServiceParams} from 'server/services/TrainsService/types/ITrainsAddInsuranceServiceParams';
import {ITrainsOrderDetails} from 'reducers/trains/order/types';
import {ITrainsDetailsApiParams} from 'server/api/TrainsApi/types/ITrainsDetailsApiParams';
import {ITrainsDetailsInfoServiceResponse} from 'server/services/TrainsService/types/ITrainsDetailsInfoServiceResponse';
import {ITrainsService} from 'server/services/TrainsService/types/ITrainsService';
import ITrainsChangeRegistrationStatusParams from 'server/api/TrainsBookingApi/types/ITrainsChangeRegistrationStatusParams';
import ITrainsGetActualizedOrderServiceParams from 'server/services/TrainsService/types/ITrainsGetActualizedOrderServiceParams';
import ITrainsDownloadBlankApiParams from 'server/api/TrainsBookingApi/types/ITrainsDownloadBlankApiParams';
import IUserData from 'server/utilities/DataStorage/UserData/types/IUserData';
import IAttributionData from 'server/utilities/DataStorage/AttributionData/types/IAttributionData';
import {
    ITrainsTestContextTokenParams,
    ITrainsTestContextTokenAnswer,
} from 'server/api/TrainsBookingApi/types/ITrainsTestContextToken';
import {ITrainsGetFirmTrainService} from 'server/services/TrainsService/types/ITrainsGetFirmTrainService/ITrainsGetFirmTrainService';
import {ITrainsGetDirectionSearchService} from 'server/services/TrainsService/types/ITrainsGetDirectionSearchService/ITrainsGetDirectionSearchService';
import {ITrainsGetTariffsService} from 'server/services/TrainsService/types/ITrainsGetTariffsService/ITrainsGetTariffsService';

import AffiliateData from 'server/utilities/DataStorage/AffiliateData/AffiliateData';
import getFirmTrainMockResponseBySlug from 'server/services/TrainsService/utilities/getFirmTrainMockResponseBySlug';

import {IDependencies} from 'server/getContainerConfig';
import {TrainsApi} from 'server/api/TrainsApi/TrainsApi';
import {TrainsBookingApi} from 'server/api/TrainsBookingApi/TrainsBookingApi';
import {TicketsTravelersNotebookService} from 'server/services/TicketsTravelersNotebookService/TicketsTravelersNotebookService';
import {GeobaseService} from 'server/services/GeobaseService/GeobaseService';
import prepareTrainsGetTariffsApiParams from 'server/services/TrainsService/prepareData/prepareTrainsGetTariffsApiParams';
import prepareTrainsGetTariffsServiceResponse from 'server/services/TrainsService/prepareData/prepareTrainsGetTariffsServiceResponse';
import prepareTrainsDetailsServiceResponse from 'server/services/TrainsService/prepareData/prepareTrainsDetailsServiceResponse';
import {TrainsOfferStorageApi} from 'server/api/TrainsOfferStorageApi/TrainsOfferStorageApi';
import prepareOrderLabelParams from 'server/services/TrainsService/prepareData/prepareOrderLabelParams';
import {RaspSuggestsApi} from 'server/api/RaspSuggestsApi/RaspSuggestsApi';
import {TestContextService} from 'server/services/TestContextService/TestContextService';
import {TrainsSearchApi} from 'server/api/TrainsSearchApi/TrainsSearchApi';

import emptyPdfText from './__mocks__/emptyPdfText';

export class TrainsService implements ITrainsService {
    private trainsApi: TrainsApi;
    private trainsSearchApi: TrainsSearchApi;
    private raspSuggestsApi: RaspSuggestsApi;
    private trainsBookingApi: TrainsBookingApi;
    private trainsOfferStorageApi: TrainsOfferStorageApi;
    private geobaseService: GeobaseService;
    private ticketsTravelersNotebookService: TicketsTravelersNotebookService;
    private readonly attributionData: IAttributionData;
    private readonly userData: IUserData;
    private readonly userInfo: TBlackboxUserInfo;
    private readonly testContextService: TestContextService;
    private readonly affiliateData: AffiliateData;
    private readonly isE2E: boolean;

    constructor({
        trainsApi,
        trainsSearchApi,
        raspSuggestsApi,
        trainsBookingApi,
        geobaseService,
        attributionData,
        userData,
        userInfo,
        ticketsTravelersNotebookService,
        trainsOfferStorageApi,
        testContextService,
        affiliateData,
        isE2E,
    }: IDependencies) {
        this.trainsApi = trainsApi;
        this.trainsSearchApi = trainsSearchApi;
        this.raspSuggestsApi = raspSuggestsApi;
        this.trainsBookingApi = trainsBookingApi;
        this.trainsOfferStorageApi = trainsOfferStorageApi;
        this.geobaseService = geobaseService;
        this.ticketsTravelersNotebookService = ticketsTravelersNotebookService;
        this.testContextService = testContextService;
        this.attributionData = attributionData;
        this.userData = userData;
        this.userInfo = userInfo;
        this.affiliateData = affiliateData;
        this.isE2E = isE2E;
    }

    getSuggests: ITrainsGetSuggestsService = async params => {
        return this.raspSuggestsApi.getSuggests(params);
    };

    getTariffs: ITrainsGetTariffsService = async params => {
        if (params.isMolly) {
            return {
                segments: [],
                querying: false,
            };
        }

        const rawTariffs = await this.trainsApi.getTariffs(
            prepareTrainsGetTariffsApiParams(params, this.userData),
        );

        return prepareTrainsGetTariffsServiceResponse(rawTariffs);
    };

    async saveOffer(
        params: ITrainsSaveOfferServiceParams,
        req: Request,
    ): Promise<ITrainsSaveOfferServiceResponse> {
        const {isTransfer, ytpReferer, ...apiParams} = params;

        const attributionData = this.attributionData;
        const userData = this.userData;
        const affiliateData = this.affiliateData;

        const userParams = await this.getUserParams(req);

        const labelParams = prepareOrderLabelParams({
            attributionData,
            affiliateData,
            userData,
            userParams,
            isTransfer,
            ytpReferer,
        });

        return this.trainsOfferStorageApi.saveOffer({
            ...apiParams,
            labelParams,
        });
    }

    /**
     * @deprecated - использовать notebookService
     */
    saveDocumentsAndTravelerToNotebook(
        params: ITrainsSaveDocumentsAndTravelerToNotebookServiceParams,
    ): Promise<void> {
        return this.ticketsTravelersNotebookService.saveDocumentsAndTravelerToNotebook(
            params,
        );
    }

    addInsurance(params: ITrainsAddInsuranceServiceParams): Promise<void> {
        return this.trainsBookingApi.addInsurance(params.orderId);
    }

    async getTrainDetails(
        params: ITrainsDetailsApiParams,
    ): Promise<ITrainsDetailsInfoServiceResponse> {
        const mockImTestContextParams =
            await this.testContextService.getTrainsMockImTestContextParamsIfNeed();

        return this.trainsApi
            .trainDetails({
                ...params,
                ...mockImTestContextParams,
                icookie: this.userData.getICookie(),
            })
            .then(prepareTrainsDetailsServiceResponse);
    }

    activePartners(params: IActivePartnersParams): Promise<ETrainsPartner[]> {
        return this.trainsApi.activePartners(params.language || LANG.RU);
    }

    logBandit(params: ITrainsLogBanditApiParams): Promise<void> {
        return this.trainsApi.logBandit(params);
    }

    async downloadBlank(
        params: ITrainsDownloadBlankApiParams,
    ): Promise<NodeJS.ReadableStream> {
        if (this.isE2E) {
            return Readable.from(emptyPdfText);
        }

        return this.trainsBookingApi.downloadBlank(params);
    }

    downloadBlankFromTrainApi({
        orderId,
        blankId,
    }: {
        orderId: string;
        blankId?: string;
    }): Promise<NodeJS.ReadableStream> {
        return this.trainsApi.downloadBlank(orderId, blankId);
    }

    /**
     * Метод получения заказа для старых заказов созданных через trainApi
     */
    async getActualizedTrainApiOrder({
        orderId,
    }: ITrainsGetActualizedOrderServiceParams): Promise<ITrainsOrderDetails> {
        return this.trainsApi.updateOrder(orderId);
    }

    async changeOrderRegistrationStatus(
        params: ITrainsChangeRegistrationStatusParams,
    ): Promise<void> {
        await this.trainsBookingApi.orderRegistrationStatusChange(params);
    }

    getFirmTrain: ITrainsGetFirmTrainService = async ({slug}) => {
        return getFirmTrainMockResponseBySlug({slug});
    };

    getDirectionSearch: ITrainsGetDirectionSearchService = async params => {
        return this.trainsSearchApi.getDirectionSearch(params);
    };

    testContextToken(
        params: ITrainsTestContextTokenParams,
    ): Promise<ITrainsTestContextTokenAnswer> {
        return this.testContextService.getTrainsTestContextToken(params);
    }

    private async getUserParams(req: Request): Promise<IUserParams> {
        const {geoId} = await this.geobaseService.getUserGeoLocation();

        return {
            ip: req.ip,
            geoId,
            userInfo: this.userInfo,
        };
    }
}
