import moment from 'moment-timezone';
import flatMap from 'lodash/flatMap';

import {GENDER_TYPE} from 'projects/trains/constants/genders';
import {TRAIN_COACH_TYPE} from 'projects/trains/constants/coachType';

import {
    ITrainsOrderDetails,
    IOrderPassenger,
    IOrderPassengerInsurance,
    IOrderTicket,
    IOrderTicketTariffInfo,
    IRefundPayment,
    RefundPaymentStatus,
} from 'reducers/trains/order/types';
import ITrainsOrderInfoResponse, {
    ITrainsOrderRefund,
    TrainsOrderRefundStatus,
} from 'server/api/TrainsBookingApi/types/ITrainsOrderInfoResponse';
import ITrainsOrderInfoPassenger from 'server/api/TrainsBookingApi/types/ITrainsOrderInfoPassenger';
import ITrainsOrderInfoInsurance from 'server/api/TrainsBookingApi/types/ITrainsOrderInfoInsurance';
import ITrainsOrderInfoTicket from 'server/api/TrainsBookingApi/types/ITrainsOrderInfoTicket';
import ITrainsOrderInfoTicketTariffInfo from 'server/api/TrainsBookingApi/types/ITrainsOrderInfoTicketTariffInfo';
import {TrainsOrderCompartmentGender} from 'server/api/TrainsBookingApi/types/TrainsOrderCompartmentGender';
import {TrainsInsuranceStatus} from 'server/api/TrainsBookingApi/types/TrainsInsuranceStatus';
import ITrainTicket from 'server/api/GenericOrderApi/types/common/service/ITrainServiceInfo/ITrainPassenger/ITrainTicket';

import prepareOrderError from 'server/services/TrainsService/prepareData/prepareOrderError';

function preparePassengerTicketTariffInfo(
    tariffInfo: ITrainsOrderInfoTicketTariffInfo | null,
): IOrderTicketTariffInfo | null {
    if (!tariffInfo) {
        return null;
    }

    return {
        minAge: 0,
        maxAge: 0,
        minAgeIncludesBirthday: false,
        maxAgeIncludesBirthday: false,
        title: tariffInfo.title,
        code: tariffInfo.code,
        needDocument: false,
        withoutPlace: tariffInfo.withoutPlace,
    };
}

function preparePassengerTicket(
    ticket: ITrainsOrderInfoTicket,
    refunds: ITrainsOrderRefund[],
    processedTickets: number[],
): IOrderTicket {
    let amount = 0;
    let type;

    for (let i = 0; i < refunds.length; i++) {
        if (processedTickets.includes(ticket.blankId)) break;

        const refund = refunds[i];

        for (let j = 0; j < refund.tickets.length; j++) {
            const ticketFromRefunds = refund.tickets[j];

            if (ticketFromRefunds.blankId === ticket.blankId) {
                amount += ticketFromRefunds.amount.value;

                type = refund.type;
            }
        }

        if (amount) {
            break;
        }
    }

    processedTickets.push(ticket.blankId);

    const places: string[] = [];
    const placesType = {
        code: '',
        description: '',
    };

    ticket.places.forEach(place => {
        places.push(place.number);
        placesType.code = place.type;
        placesType.description = place.typeText;
    });

    return {
        blankId: String(ticket.blankId),
        rzhdStatus: ticket.rzhdStatus,
        places,
        placesType,
        tariffInfo: preparePassengerTicketTariffInfo(ticket.tariffInfo),
        rawTariffTitle: ticket.rawTariffTitle,
        bookedTariffCode: ticket.bookedTariffCode,
        refund: type ? {amount, type} : null,
        pending: ticket.pending,
        isRefunding: ticket.refunding,
        amount: ticket.amount.value,
        payment: {
            amount: ticket.payment.amount.value,
            fee: ticket.payment.fee.value,
            beddingAmount: ticket.payment.beddingAmount.value,
        },
        canReturnTill: ticket.canReturnTill,
        canChangeElectronicRegistrationTill:
            ticket.canChangeElectronicRegistrationTill,
    };
}

export function preparePassengerInsurance(
    insurance: ITrainsOrderInfoInsurance | null,
    insuranceStatus: TrainsInsuranceStatus,
): IOrderPassengerInsurance {
    if (!insurance) {
        return {
            ordered: false,
            refunded: false,
            compensation: 0,
            amount: 0,
            compensationVariants: [],
            operationId: null,
        };
    }

    return {
        ordered: insuranceStatus === TrainsInsuranceStatus.CHECKED_OUT,
        refunded: false,
        compensation: insurance.compensation.value,
        amount: insurance.amount.value,
        compensationVariants: [],
        operationId: null,
    };
}

function preparePassenger(
    passenger: ITrainsOrderInfoPassenger,
    insuranceStatus: TrainsInsuranceStatus,
    refunds: ITrainsOrderRefund[],
    processedTickets: number[],
): IOrderPassenger {
    return {
        tickets: passenger.tickets.map(ticket =>
            preparePassengerTicket(ticket, refunds, processedTickets),
        ),
        customerId: String(passenger.customerId),
        docId: passenger.docId,
        docType: passenger.docType,
        citizenship: passenger.citizenship,
        firstName: passenger.firstName,
        lastName: passenger.lastName,
        patronymic: passenger.patronymic,
        birthDate: passenger.birthDate,
        age: passenger.age,
        sex: passenger.sex,
        insurance: preparePassengerInsurance(
            passenger.insurance,
            insuranceStatus,
        ),
        isNonRefundableTariff: passenger.isNonRefundableTariff,
    };
}

export function prepareCompartmentGender(
    gender: TrainsOrderCompartmentGender | null,
): GENDER_TYPE {
    if (gender === TrainsOrderCompartmentGender.FEMALE) {
        return GENDER_TYPE.FEMALE;
    }

    if (gender === TrainsOrderCompartmentGender.MALE) {
        return GENDER_TYPE.MALE;
    }

    // Если:
    // gender === TrainsOrderCompartmentGender.MIXED
    // gender === null
    return GENDER_TYPE.MIXED;
}

function prepareOrderRefundPaymentStatus(
    status: TrainsOrderRefundStatus,
): RefundPaymentStatus {
    if (status === TrainsOrderRefundStatus.SUCCESS) {
        return RefundPaymentStatus.DONE;
    }

    if (status === TrainsOrderRefundStatus.FAIL) {
        return RefundPaymentStatus.FAILED;
    }

    return RefundPaymentStatus.NEW;
}

function prepareOrderRefundPayments(
    refunds: ITrainsOrderRefund[],
): IRefundPayment[] {
    return refunds.map(({tickets, paymentReceiptUrl, createdAt, status}) => {
        return {
            blankIds: tickets.map(({blankId}) => String(blankId)),
            paymentRefundReceiptUrl: paymentReceiptUrl || '',
            refundCreatedAt: createdAt,
            insuranceIds: [], // TODO TRAINS
            refundPaymentStatus: prepareOrderRefundPaymentStatus(status),
        };
    });
}

export function calculateTicketsStatusFreezesAt(
    tickets: ITrainsOrderInfoTicket[] | ITrainTicket[],
): string | null {
    let ticketsStatusFreezesAt: string | null = null;

    for (const ticket of tickets) {
        const {canChangeElectronicRegistrationTill} = ticket;

        if (!canChangeElectronicRegistrationTill) {
            continue;
        }

        if (
            !ticketsStatusFreezesAt ||
            moment(canChangeElectronicRegistrationTill).isBefore(
                moment(ticketsStatusFreezesAt),
            )
        ) {
            ticketsStatusFreezesAt = canChangeElectronicRegistrationTill;
        }
    }

    return ticketsStatusFreezesAt;
}

export default function prepareOrderResponse(
    order: ITrainsOrderInfoResponse,
): ITrainsOrderDetails {
    const processedTickets: number[] = [];

    return {
        uid: order.id,
        orderNumber: String(order.prettyId),
        partnerOrderNumber: order.reservationNumber,
        status: order.status,
        passengers: order.passengers.map(passenger =>
            preparePassenger(
                passenger,
                order.insuranceStatus,
                order.refunds || [],
                processedTickets,
            ),
        ),
        carType: order.carType as TRAIN_COACH_TYPE,
        trainName: order.trainInfo.brandTitle,
        carNumber: order.carNumber,
        trainNumber: order.trainInfo.trainNumber,
        trainTicketNumber: order.trainInfo.trainTicketNumber,
        coachOwner: order.coachOwner,
        companyTitle: order.companyTitle,
        isSuburban: order.trainInfo.suburban,
        twoStorey: order.twoStorey,
        startStation:
            order.trainInfo.startStationTitle ||
            order.trainInfo.startSettlementTitle,
        endStation: order.trainInfo.endSettlementTitle,
        stationFrom: order.stationFrom,
        stationTo: order.stationTo,
        departure: order.departure,
        arrival: order.arrival,
        reservedTo: order.reservedTo,
        ticketsStatusFreezesAt: calculateTicketsStatusFreezesAt(
            flatMap(order.passengers, p => p.tickets),
        ),
        rebookingAvailable: order.rebookingAvailable,
        compartmentGender: prepareCompartmentGender(order.compartmentGender),
        insuranceStatus: order.insuranceStatus,
        partner: order.partner,
        error: prepareOrderError(order.error),
        warnings: order.warnings ?? [],
        specialNotice: order.specialNotice,
        isOnlyFullReturnPossible: order.onlyFullReturnPossible,
        paymentUrl: order.paymentUrl,
        paymentCode: order.paymentCode,
        paymentReceiptUrl: order.paymentReceiptUrl,
        refundPayments: prepareOrderRefundPayments(order.refunds || []),
        userInfo: {email: order.customerEmail, phone: order.customerPhone},
        isFullRefundPossible: order.isFullRefundPossible,
    };
}
