import keyBy from 'lodash/keyBy';

import {ITrainsDetailsApiResponse} from 'server/api/TrainsApi/types/ITrainsDetailsApiResponse';
import {ITrainsDetailsInfoServiceResponse} from 'server/services/TrainsService/types/ITrainsDetailsInfoServiceResponse';

import shouldUseServiceClassesAsCoachCategories from 'server/services/TrainsService/prepareData/utilities/shouldUseServiceClassesAsCoachCategories';

import * as i18nBlock from 'i18n/trains-error-page-order';

import {prepareTrainsDetailsCoach} from 'server/services/TrainsService/prepareData/prepareTrainsDetailsCoach';

export default function prepareTrainsDetailsServiceResponse(
    apiResponse: ITrainsDetailsApiResponse,
): ITrainsDetailsInfoServiceResponse {
    const {trainDetails, errors, partnerError} = apiResponse;

    const coaches = trainDetails?.coaches;

    if (!trainDetails || !coaches || coaches.length === 0) {
        return {
            trainDetails: null,
            errors: true,
            partnerError: {
                code: 'COMMON',
                description: i18nBlock.ufsDashErrorDashCodeDash5060(),
            },
        };
    }

    const useServiceClasses = shouldUseServiceClassesAsCoachCategories(coaches);

    const preparedCoaches = coaches
        .map((coach, index) =>
            prepareTrainsDetailsCoach({
                coach,
                index,
                useServiceClasses,
            }),
        )
        .sort((a, b) => Number(a.number) - Number(b.number));

    return {
        trainDetails: {
            ...trainDetails,
            coaches: preparedCoaches,
            schemas: keyBy(trainDetails.schemas || [], 'id'),
        },
        errors: Boolean(errors),
        partnerError: partnerError || null,
    };
}
