import moment from 'moment-timezone';

import {ITrainsGetTariffsApiResponse} from 'server/api/TrainsApi/types/ITrainsGetTariffsApi/ITrainsGetTariffsApi';
import {ITrainsTariffApiSegment} from 'server/api/TrainsApi/types/ITrainsGetTariffsApi/models';
import {ITrainsGetTariffsServiceResponse} from 'server/services/TrainsService/types/ITrainsGetTariffsService/ITrainsGetTariffsService';

import {ROBOT_DATETIME_UTC} from 'utilities/dateUtils/formats';
import {CHAR_EM_DASH, CHAR_NBSP} from 'utilities/strings/charCodes';

const dashRegexp = new RegExp('(\\s[-—])(\\s)');
const shortWordRegexp = new RegExp('([^-—])((\\s)(\\S{1,5}))$');

/**
 * Подставляет в заголовок неразрывные пробелы для двух кейсов:
 *   - Подставлять неразрывный пробел перед короткими словами в конце заголовка
 *   - Подставлять неразрывный пробел перед длинным тире
 * Задача: RASPFRONT-2004 - Оформление и типографика в сниппете
 *
 * @param title - строка заголовка сегмента
 */
function patchTitle(title: string | undefined): string {
    let patchedTitle = title || '';

    if (patchedTitle) {
        patchedTitle = patchedTitle
            .replace(dashRegexp, `${CHAR_NBSP}${CHAR_EM_DASH}$2`)
            .replace(shortWordRegexp, `$1${CHAR_NBSP}$4`);
    }

    return patchedTitle;
}

function patchSegments(
    segments: ITrainsTariffApiSegment[],
): ITrainsTariffApiSegment[] {
    return segments.map(segment => {
        return {
            ...segment,
            title: patchTitle(segment.title),
        };
    });
}

/**
 * Вернёт признак фильтрации тарифа
 * Фильтруем тариф если отправление в прошлом или до него осталось меньше часа
 *
 * @param segment - данные сегмента
 */
function shouldRemoveTariffs(segment: ITrainsTariffApiSegment): boolean {
    const departure = moment(segment.departure, ROBOT_DATETIME_UTC);
    const removeTariffsBefore = moment().add(1, 'hour');

    return departure.isBefore(removeTariffsBefore);
}

export default function prepareTrainsGetTariffsServiceResponse(
    apiResponse: ITrainsGetTariffsApiResponse,
): ITrainsGetTariffsServiceResponse {
    const filteredSegments = apiResponse.segments.filter(
        segment => !shouldRemoveTariffs(segment),
    );

    const patchedSegments = patchSegments(filteredSegments);

    return {
        ...apiResponse,
        segments: patchedSegments,
    };
}
