import {ServerOptions, WebSocket, WebSocketServer} from 'ws';

import {SECOND} from 'utilities/dateUtils/constants';

import {TRequestEvent} from 'server/utilities/TestRequestManager/types/events';
import {NextFunction, Request, Response} from '@yandex-data-ui/core/lib/types';

export interface IClientSocket {
    sessionKey: string;
    socket: WebSocket;
    pageToken: string | null;
}

export class ApiRequestChannel extends WebSocketServer {
    clientSockets = new Set<IClientSocket>();

    constructor(options?: ServerOptions, callback?: () => void) {
        super(options, callback);

        // Чтобы сокет не уходил в состояние idle через минуту и nginx не убивал его
        setInterval(() => {
            for (const {socket} of this.clientSockets) {
                socket.send('ping');
            }
        }, 5 * SECOND);
    }

    addClientSocket(clientSocket: IClientSocket): void {
        this.clientSockets.add(clientSocket);

        clientSocket.socket.addEventListener('close', () => {
            this.clientSockets.delete(clientSocket);
        });
    }

    sendRequestEvent(
        targetSessionKey: string,
        targetPageToken: string | null,
        event: TRequestEvent,
    ): void {
        for (const {sessionKey, pageToken, socket} of this.clientSockets) {
            // Хотим отправять клиентам информацию только об их собственных запросах
            if (
                sessionKey === targetSessionKey &&
                (pageToken === null || pageToken === targetPageToken)
            ) {
                socket.send(JSON.stringify(event));
            }
        }
    }
}

export const apiRequestsChannel = new ApiRequestChannel({noServer: true});

export const apiRequestChannelMiddleware = (
    req: Request,
    res: Response,
    next: NextFunction,
): void => {
    req.apiRequestsChannel = apiRequestsChannel;

    next();
};
