const App = require('@yandex-data-ui/core');
const expressTld = require('@yandex-int/express-tld');

const {JAEGER_TRACER_OPTIONS} = require('./constants/common');

const initTracer = require('./utilities/jaeger/initTracer');

const config = require('../.config');
const csrfMiddleware = require('./middlewares/csrfMiddleware');
const expressYandexUidMiddleware = require('./middlewares/expressYandexUidMiddleware');
const deviceTypeMiddleware = require('./middlewares/deviceTypeMiddleware');
const featuresMiddleware = require('./middlewares/featuresMiddleware');
const expressVaryMiddleware = require('./middlewares/express-vary');
const userInfoMiddleware = require('./middlewares/userInfoMiddleware');
const userNetworkMiddleware = require('./middlewares/userNetworkMiddleware');
const containerMiddleware = require('./middlewares/containerMiddleware');
const experimentsMiddleware = require('./middlewares/experimentsMiddleware');
const cookieSessionMiddleware = require('./middlewares/cookieSessionMiddleware');
const sessionTravelUidMiddleware = require('./middlewares/sessionTravelUidMiddleware');
const userUnreadMailCountMiddleware = require('./middlewares/userUnreadMailCountMiddleware');
const aviaFeaturesMiddleware = require('./middlewares/aviaFeaturesMiddleware');
const userLanguageMiddleware = require('./middlewares/userLanguageMiddleware');
const userNationalVersionMiddleware = require('./middlewares/userNationalVersionMiddleware');
const aviaCookieMiddleware = require('./middlewares/aviaCookieMiddleware');
const utmMiddleware = require('./middlewares/utmMiddleware');
const jaegerSpanMiddleware = require('./middlewares/jaegerSpanMiddleware');
const platformMiddleware = require('./middlewares/platformMiddleware').default;
const isTravelAppMiddleware =
    require('./middlewares/isTravelAppMiddleware').default;
const fakePlatformMiddleware = require('./middlewares/fakePlatformMiddleware');
const nativeAppStateMiddleware = require('./middlewares/nativeAppStateMiddleware');
const {
    nativeAppVersionMiddleware,
} = require('./middlewares/nativeAppVersionMiddleware');
const forceMobileVersionMiddleware = require('./middlewares/forceMobileVersionMiddleware');
const botCheckMiddleware = require('./middlewares/botCheckMiddleware');
const incompatibleMiddleware = require('./middlewares/incompatibleMiddleware');
const decodeUrlMiddleware = require('./middlewares/decodeUrlMiddleware');
const requestInfoMiddleware =
    require('./middlewares/requestInfoMiddleware').default;
const whiteLabelMiddleware =
    require('./middlewares/whiteLabelMiddleware').default;
const {
    apiRequestsChannel,
    apiRequestChannelMiddleware,
} = require('./sockets/apiRequestsChannel');

module.exports = () => {
    const app = new App({name: 'ya-travel'});

    initTracer();

    // Надо, чтобы было первым, чтобы засечь время в начале запроса
    app.useMiddleware({
        stage: 'beforeAuth',
        fn: requestInfoMiddleware,
    });

    app.useMiddleware({
        stage: 'beforeAuth',
        fn: csrfMiddleware,
    });

    app.useMiddleware({
        stage: 'afterAuth',
        fn: userLanguageMiddleware,
    });

    app.useMiddleware({
        stage: 'afterAuth',
        fn: userNationalVersionMiddleware,
    });

    app.registerMiddleware({
        name: 'featuresMiddleware',
        fn: featuresMiddleware,
    });

    app.registerMiddleware({
        name: 'expressVaryMiddleware',
        fn: expressVaryMiddleware,
    });

    app.registerMiddleware({
        name: 'userInfoMiddleware',
        fn: userInfoMiddleware,
    });

    app.registerMiddleware({
        name: 'userNetworkMiddleware',
        fn: userNetworkMiddleware,
    });

    app.registerMiddleware({
        name: 'experimentsMiddleware',
        fn: experimentsMiddleware,
    });

    app.registerMiddleware({
        name: 'jaegerApiMiddleware',
        fn: jaegerSpanMiddleware(JAEGER_TRACER_OPTIONS.API_SERVICE_NAME),
    });

    app.registerMiddleware({
        name: 'jaegerPageMiddleware',
        fn: jaegerSpanMiddleware(JAEGER_TRACER_OPTIONS.PAGE_SERVICE_NAME),
    });

    app.registerMiddleware({
        name: 'decodeUrlMiddleware',
        fn: decodeUrlMiddleware,
    });

    app.registerMiddleware({
        name: 'incompatibleMiddleware',
        fn: incompatibleMiddleware,
    });

    app.useMiddleware({
        stage: 'beforeAuth',
        fn: deviceTypeMiddleware,
    });

    app.useMiddleware({
        stage: 'beforeAuth',
        fn: expressYandexUidMiddleware,
    });

    app.useMiddleware({
        stage: 'beforeAuth',
        fn: expressTld(),
    });

    app.useMiddleware({
        stage: 'afterAuth',
        fn: cookieSessionMiddleware(app.config),
    });

    app.useMiddleware({
        stage: 'afterAuth',
        fn: sessionTravelUidMiddleware,
    });

    app.useMiddleware({
        stage: 'afterAuth',
        fn: userUnreadMailCountMiddleware,
    });

    app.useMiddleware({
        stage: 'afterAuth',
        fn: aviaFeaturesMiddleware,
    });

    app.useMiddleware({
        stage: 'afterAuth',
        fn: containerMiddleware,
    });

    app.useMiddleware({
        stage: 'afterAuth',
        fn: aviaCookieMiddleware,
    });

    app.useMiddleware({
        stage: 'afterAuth',
        fn: utmMiddleware,
    });

    app.useMiddleware({
        stage: 'afterAuth',
        fn: platformMiddleware,
    });

    app.useMiddleware({
        stage: 'afterAuth',
        fn: isTravelAppMiddleware,
    });

    if (process.env.APP_ENV !== 'production') {
        app.useMiddleware({
            stage: 'afterAuth',
            fn: fakePlatformMiddleware,
        });

        app.useMiddleware({
            stage: 'afterAuth',
            fn: forceMobileVersionMiddleware,
        });
    }

    app.useMiddleware({
        stage: 'afterAuth',
        fn: nativeAppStateMiddleware,
    });

    app.useMiddleware({
        stage: 'afterAuth',
        fn: nativeAppVersionMiddleware,
    });

    app.useMiddleware({
        stage: 'afterAuth',
        fn: botCheckMiddleware,
    });

    app.useMiddleware({
        stage: 'afterAuth',
        fn: whiteLabelMiddleware,
    });

    if (config.env !== 'production') {
        app.registerWSServer('apiRequestsChannel', apiRequestsChannel);
        app.useMiddleware({
            stage: 'afterAuth',
            fn: apiRequestChannelMiddleware,
        });
    }

    return app;
};
