import cuid from 'cuid';

import {EApiEntry} from 'types/EApiEntry';

import {IDependencies} from 'server/getContainerConfig';

import {RestApiClient} from '../RestApiClient';
import {IRequestConfig} from '../HttpClient/IHttpClient';
import AviaBackendApiError from './AviaBackendApiError';
import {
    IAviaBackendApiClient,
    IAviaBackendRequest,
} from './IAviaBackendApiClient';

interface IApiResponse {
    status: string;
    [key: string]: any;
}

export interface IResponse<T> extends IApiResponse {
    [key: string]: string | T | undefined;
    data?: T & IApiResponse;
}

const DEFAULT_TIMEOUT = 5000;

export class AviaGatewayBackendApiClient
    extends RestApiClient
    implements IAviaBackendApiClient
{
    private yandexuid: string;
    private passportid: string;
    private readonly nationalVersion: string;

    constructor({
        httpClient,
        getApiHost,
        getSrcParams,
        logger,
        requestId,
        nationalVersion,
        sendClickHouseStats,
        blackbox,
        yandexuid,
        rootSpan,
    }: IDependencies) {
        super({
            baseURL: `${getApiHost(EApiEntry.AVIA_GATEWAY)}/backend`,
            httpClient,
            logger,
            requestId,
            sendClickHouseStats,
            rootSpan,
        });

        this.srcParams = getSrcParams(EApiEntry.AVIA_GATEWAY);
        this.yandexuid = yandexuid;
        this.passportid = blackbox.uid;
        this.nationalVersion = nationalVersion;
    }

    async request<T>(
        aviaBackendRequest: IAviaBackendRequest,
    ): Promise<T | null> {
        const params = this.extractQueryParams(aviaBackendRequest.name);

        return this.sendAndValidateRequest<T>(
            {
                params,
                headers: {
                    'user-agent': 'travel-frontend',
                    'content-type': 'application/json',
                },
                timeout: DEFAULT_TIMEOUT,
            },
            aviaBackendRequest,
        ).catch(() => {
            /** ignore error */
            return null;
        });
    }

    private async sendAndValidateRequest<T>(
        options: IRequestConfig,
        aviaBackendRequest: IAviaBackendRequest,
    ): Promise<T> {
        let json;

        try {
            json = await this.post<IResponse<T>>(
                '/',
                [aviaBackendRequest],
                options,
            );
        } catch (err) {
            throw new AviaBackendApiError(err, aviaBackendRequest, '');
        }

        // Даже с кодом 200 может быть ошибка
        if (!json || !json.status || !json.data) {
            throw new AviaBackendApiError(
                'Внутренняя ошибка API',
                aviaBackendRequest,
                JSON.stringify(json),
            );
        }

        // По идеи всегда должен быть код ответа не 200, но мало ли
        if (json.status !== 'success') {
            throw new AviaBackendApiError(
                'Ошибка обращения к API',
                aviaBackendRequest,
                JSON.stringify(json),
            );
        }

        const data = json.data[0];

        if (data && data.status === 'error') {
            throw new AviaBackendApiError(
                'Ошибка обращения к API',
                aviaBackendRequest,
                JSON.stringify(data),
            );
        }

        if (this.checkError<T & IApiResponse>(data)) {
            this.logger.logError(
                'Ошибка вложенного запроса к API',
                new AviaBackendApiError(
                    'Ошибка вложенного запроса к API',
                    aviaBackendRequest,
                    JSON.stringify(data),
                ),
            );
        }

        return data;
    }

    private checkError<T>(
        json: T & IApiResponse,
        field?: string | number | symbol,
    ): boolean {
        if (!json) {
            return false;
        }

        if (Array.isArray(json)) {
            let hasError = false;

            for (let i = 0; i < json.length; i++) {
                if (this.checkError(json[i])) {
                    json[i] = undefined;
                    hasError = true;
                }
            }

            return hasError;
        }

        if (typeof json === 'object') {
            if (field && json.status === 'error') {
                return true;
            }

            const fields = Object.keys(json) as (keyof typeof json)[];
            let hasError = false;

            fields.forEach(jsonField => {
                const fieldRes = json[jsonField];

                if (!fieldRes) {
                    return;
                }

                if (this.checkError(fieldRes, jsonField)) {
                    // Если вложенный ресурс содержит ошибку
                    // исключаем его из ответа
                    json[jsonField] = undefined;
                    hasError = true;
                }
            });

            return hasError;
        }

        return false;
    }

    private extractQueryParams(nameHandler: string): object {
        return {
            national_version: this.nationalVersion,
            lang: 'ru',

            // ID запроса нужен для логирования
            cuid: cuid(),

            // Инфа о пользователя нужна для логирования
            yandexuid: this.yandexuid,
            passportuid: this.passportid,

            name: nameHandler,
        };
    }
}
