import {ParsedQuery} from 'query-string';
import {
    pick,
    omit,
    mapValues,
    isEqual,
    identity,
    pickBy,
    isEmpty,
} from 'lodash';

import {attributionFields} from 'server/utilities/DataStorage/AttributionData/constants/attributionFields';
import mapFieldToQuery from 'server/utilities/DataStorage/AttributionData/constants/mapFieldToQuery';
import {AFFILIATE_COOKIE_NAME} from 'constants/affiliate/cookieName';
import {INSIGNIFICANT_PARAMS} from 'server/utilities/DataStorage/AttributionData/constants/insignificantParams';
import {EAffiliateQueryParams} from 'constants/affiliate/queryParams';

import {
    ECookieName,
    TCookiesMeta,
} from 'server/utilities/DataStorage/AttributionData/types/ICookie';
import IAttributionDataValues from 'types/attribution/IAttributionDataValues';
import IAttributionUTMDataValues from 'types/attribution/IAttributionUTMDataValues';
import IAttributionData from 'server/utilities/DataStorage/AttributionData/types/IAttributionData';
import {ICookies} from '@yandex-data-ui/core/lib/types';
import {TNormalizedQuery} from 'server/utilities/DataStorage/AttributionData/types/TNormalizedQuery';
import IAttributionUaas from 'server/utilities/DataStorage/AttributionData/types/IAttributionUaas';
import EAttributionField from 'types/attribution/EAttributionField';
import {TSetCookie} from 'types/TSetCookie';

import getOrganicSource from 'server/utilities/utm/utils/getOrganicSource';
import {ILogger} from 'server/utilities/Logger';

import {IDependencies} from 'server/getContainerConfig';

type TAttributionDataParams = Pick<
    IDependencies,
    'logger' | 'query' | 'cookies' | 'setCookie' | 'referrer' | 'testBuckets'
>;

/**
 * Хранилище липких данных о пользователе
 */
export default class AttributionData implements IAttributionData {
    static readonly COOKIES_META: TCookiesMeta = {
        /**
         * session
         */
        [ECookieName.TRAVEL_ATTRIBUTION_SESSION]: {
            expires: 0,
            fields: attributionFields,
        },
    };

    private readonly query: ParsedQuery = {};
    private readonly cookies: ICookies = {};
    private readonly setCookie: TSetCookie;
    private readonly logger: ILogger;
    private readonly referrer: string | undefined;
    private readonly testBuckets: string | undefined;

    private data: IAttributionDataValues = {};

    constructor({
        logger,
        query,
        cookies,
        setCookie,
        referrer,
        testBuckets,
    }: TAttributionDataParams) {
        this.logger = logger;
        this.query = query;
        this.cookies = cookies;
        this.setCookie = setCookie;
        this.referrer = referrer;
        this.testBuckets = testBuckets;
    }

    getUtm(): IAttributionUTMDataValues {
        const {utmSource, utmMedium, utmCampaign, utmTerm, utmContent} =
            this.data;

        return {
            utmSource,
            utmMedium,
            utmCampaign,
            utmTerm,
            utmContent,
        };
    }

    getSerpReqId(): string | undefined {
        return this.data.serpReqId;
    }

    getSerpUuid(): string | undefined {
        return this.data.serpUuid;
    }

    getSerpTestId(): string | undefined {
        return this.data.serpTestId;
    }

    getGclid(): string | undefined {
        return this.data.gclid;
    }

    getYclid(): string | undefined {
        return this.data.yclid;
    }

    getFbclid(): string | undefined {
        return this.data.fbclid;
    }

    getYtpReferer(): string | undefined {
        return this.data.ytpReferer;
    }

    getFrom(): string | undefined {
        return this.data.from;
    }

    getUaas(): IAttributionUaas {
        const {testBuckets} = this;

        return {
            testBuckets,
        };
    }

    getWizardFlags(): string | undefined {
        return this.data.wizardFlags;
    }

    getWizardredirkey(): string | undefined {
        return this.data.wizardRedirKey;
    }

    getUserRegion(): string | undefined {
        return this.data.userRegion;
    }

    updateCookiesAndFillData(): void {
        // Удаляем nullable поля
        const referrerDataValues = pickBy(this.loadDataValuesFromReferrer());
        const cookiesDataValues = pickBy(this.loadDataValuesFromCookies());
        const queryDataValues = pickBy(this.loadDataValuesFromQuery());

        // Влияют только на логику записи/очистки кук, но не на содержимое
        const significantQueryValues =
            this.filterInsignificantParams(queryDataValues);
        const significantCookiesDataValues =
            this.filterInsignificantParams(cookiesDataValues);

        // Не мержим queryDataValues с cookiesDataValues, чтобы не создавать конфликтные наборы (метки рекламы вместе с метками колдунов)
        const dataValues = {
            ...pickBy(referrerDataValues, identity),
            ...(isEmpty(significantQueryValues)
                ? cookiesDataValues
                : queryDataValues),
        };

        this.clearAttributionIfNeeded(queryDataValues);
        this.clearAffiliateParamsIfNeeded(
            significantQueryValues,
            significantCookiesDataValues,
        );

        Object.values(ECookieName).forEach(cookieName => {
            this.saveCookie(cookieName, dataValues);
        });

        this.data = dataValues;
    }

    /*
     * Значимым считаем только последний источник трафика, поэтому
     * в случае нового источника трафика, предыдущие метки сбрасываем
     *
     * Например: Пользователь перешел с аффилиатки => сохранили параметры =>
     * => тот же пользователь перешел второй раз с поиска =>
     * => сбрасываем параметры, чтобы не платить за них
     */
    private clearAffiliateParamsIfNeeded(
        queryDataValues: Partial<IAttributionDataValues>,
        cookiesDataValues: Partial<IAttributionDataValues>,
    ): void {
        if (isEmpty(queryDataValues)) {
            return;
        }

        if (
            Object.keys(queryDataValues).some(
                key =>
                    queryDataValues[key as EAttributionField] !==
                    cookiesDataValues[key as EAttributionField],
            )
        ) {
            this.setCookie(AFFILIATE_COOKIE_NAME, '', {expires: new Date(0)});
        }
    }

    private filterInsignificantParams(
        params: IAttributionDataValues,
    ): IAttributionDataValues {
        return omit(params, INSIGNIFICANT_PARAMS);
    }

    private clearAttributionIfNeeded(
        queryDataValues: Partial<IAttributionDataValues>,
    ): void {
        if (
            this.query[EAffiliateQueryParams.AFFILIATE_CLID] ||
            this.query[EAffiliateQueryParams.AFFILIATE_VID] ||
            this.query[EAffiliateQueryParams.ADMITAD_UID] ||
            this.query[EAffiliateQueryParams.REFERRAL_PARTNER_REQUEST_ID]
        ) {
            Object.values(ECookieName).forEach(cookieName => {
                this.saveCookie(cookieName, queryDataValues);
            });
        }
    }

    private loadDataValuesFromCookies(): IAttributionDataValues {
        return Object.values(ECookieName).reduce((acc, cookieName) => {
            return {
                ...acc,
                ...this.parseCookie(cookieName),
            };
        }, {} as IAttributionDataValues);
    }

    private loadDataValuesFromReferrer(): IAttributionDataValues {
        const referrer = this.referrer;
        const utmSource = getOrganicSource(referrer);

        return {
            utmSource,
        };
    }

    private loadDataValuesFromQuery(): IAttributionDataValues {
        return attributionFields.reduce<IAttributionDataValues>(
            (acc, field) => {
                const queryParamNames = mapFieldToQuery[field];

                acc[field] = this.getFirstExistingValueFromQuery(
                    this.query,
                    queryParamNames,
                );

                return acc;
            },
            {},
        );
    }

    private getFirstExistingValueFromQuery(
        query: ParsedQuery,
        queryParamNames: string[],
    ): string | undefined {
        return this.normalizeValue(
            queryParamNames.map(name => query[name]).find(identity),
        );
    }

    /**
     * Сохраняет в куках зафиксированные в COOKIES_META поля с зафиксированным expires
     */
    private saveCookie(
        cookieName: ECookieName,
        dataValues: IAttributionDataValues,
    ): void {
        const meta = AttributionData.COOKIES_META[cookieName];
        const {expires, fields} = meta;

        const normalizedDataValues = this.normalize(dataValues);
        const suitableDataValues = pick(normalizedDataValues, fields);

        const currentCookieValue = this.parseCookie(cookieName);

        if (isEqual(suitableDataValues, currentCookieValue)) {
            return;
        }

        this.setCookie(cookieName, JSON.stringify(suitableDataValues), {
            expires: expires > 0 ? new Date(Date.now() + expires) : undefined,
        });
    }

    private parseCookie(cookieName: ECookieName): IAttributionDataValues {
        try {
            const cookieValue = this.cookies[cookieName];

            if (!cookieValue) {
                return {};
            }

            return JSON.parse(cookieValue);
        } catch (e) {
            this.logger.logWarn(
                `Failed load AttributionData from cookie: ${cookieName}`,
            );

            return {};
        }
    }

    /**
     * убираем массивы и null из параметров
     * https://st.yandex-team.ru/TRAVELFRONT-3334
     */
    private normalize(query: IAttributionDataValues): TNormalizedQuery {
        return mapValues(query, val => this.normalizeValue(val));
    }

    private normalizeValue(
        value: string | string[] | null | undefined,
    ): string | undefined {
        if (!value) {
            return undefined;
        }

        if (Array.isArray(value)) {
            this.logger.logWarn('Utm label contains array');

            return value[0];
        }

        return value;
    }
}
