import camelcaseKeys from 'camelcase-keys';
import {AxiosError} from 'axios';
import {Span, Tags, FORMAT_HTTP_HEADERS} from 'opentracing';
import StackTrace from 'stacktrace-js';
import omit from 'lodash/omit';

import {CommonHeaders} from 'server/constants/headers';
import {JAEGER_TRACER_OPTIONS} from 'server/constants/common';

const DEFAULT_UNKNOWN_AXIOS_ERROR_CODE = 598;
const DEFAULT_EXCPETION_ERROR_CODE = 599;
const {API_SERVICE_NAME} = JAEGER_TRACER_OPTIONS;

import {unknownToErrorOrUndefined} from 'utilities/error';

import {
    IHttpClient,
    IRequestConfig,
    IResponse,
    IErrorResponse,
} from './HttpClient/IHttpClient';
import {ILogger} from './Logger';

interface IRestApiClientConfig {
    baseURL: string;
    requestId: string;
    logger: ILogger;
    httpClient: IHttpClient;
    rootSpan?: Span;
    sendClickHouseStats?: (data: {[name: string]: string | number}) => void;
}

export class RestApiClient {
    protected httpClient: IHttpClient;

    protected logger: ILogger;

    protected sendClickHouseStats?: (data: {
        [name: string]: string | number;
    }) => void;

    protected srcParams?: Record<string, string>;

    protected baseURL: string;

    protected requestId: string;

    protected timeout = 25000;

    protected rootSpan?: Span;

    protected constructor({
        baseURL,
        requestId,
        logger,
        httpClient,
        sendClickHouseStats,
        rootSpan,
    }: IRestApiClientConfig) {
        this.baseURL = baseURL;
        this.requestId = requestId;
        this.logger = logger;
        this.httpClient = httpClient;
        this.sendClickHouseStats = sendClickHouseStats;
        this.rootSpan = rootSpan;

        logger.addExtra({api: this.getClassName()});
    }

    protected getClassName(): string {
        return this.constructor.name;
    }

    protected interceptRequest(_request: IRequestConfig): void | Promise<void> {
        return;
    }

    protected get<T>(path: string, options?: IRequestConfig): Promise<T> {
        return this.fetch<T>('GET', path, null, options);
    }

    protected delete<T>(path: string, options?: IRequestConfig): Promise<T> {
        return this.fetch<T>('DELETE', path, null, options);
    }

    protected post<T>(
        path: string,
        body?: any,
        options?: IRequestConfig,
    ): Promise<T> {
        return this.fetch<T>('POST', path, body, options);
    }

    protected put<T>(
        path: string,
        body?: any,
        options?: IRequestConfig,
    ): Promise<T> {
        return this.fetch<T>('PUT', path, body, options);
    }

    protected patch<T>(
        path: string,
        body?: any,
        options?: IRequestConfig,
    ): Promise<T> {
        return this.fetch<T>('PATCH', path, body, options);
    }

    private async fetch<T>(
        method: string,
        path: string,
        body?: any,
        options?: IRequestConfig,
    ): Promise<T> {
        const spanName = this.getSpanName();
        const fullPath = this.getFullPath(path, options);
        const request = this.createRequest(fullPath, method, body, options);

        await this.interceptRequest(request);

        let data = await this.trace(request, spanName);

        if (data && options && options.convertResponseKeysToCamelCase) {
            if (options.withStatus) {
                return {
                    data: camelcaseKeys(data.data, {deep: true}),
                    status: data.status,
                } as any;
            }

            data = camelcaseKeys(data, {deep: true});
        }

        return data;
    }

    private async trace(
        request: IRequestConfig,
        spanName?: string,
    ): Promise<any> {
        const start = Date.now();
        let response;
        const span = this.startTrace(request, spanName);

        try {
            this.logRequestBefore(request);
            response = await this.httpClient.request(request);
            span?.finish();
            this.logRequestSuccess(request, response, Date.now() - start);
            this.sendTelemetry(request, response, Date.now() - start);

            if (request.withStatus) {
                return {
                    data: response.data,
                    status: response.status,
                };
            }

            return response.data;
        } catch (ex) {
            span?.setTag(Tags.ERROR, true);
            span?.finish();

            const error = unknownToErrorOrUndefined(ex);

            if (error) {
                this.logRequestFailure(
                    request,
                    response,
                    error,
                    Date.now() - start,
                );
                this.sendFaultTelemetry(request, error, Date.now() - start);
            }

            throw ex;
        }
    }

    private logRequestBefore(request: IRequestConfig): void {
        this.logger.log(this.prepareMessage(request, 'request started'), {
            method: request.method,
            url: request.url,
        });
    }

    private prepareMessage(
        {method, url}: IRequestConfig,
        message: string,
        status?: number,
    ): string {
        return `${
            status ? `${status} - ` : ''
        }${message} ${this.getClassName()} : ${method} ${url}`;
    }

    private logRequestSuccess(
        request: IRequestConfig,
        response: IResponse<any>,
        duration: number,
    ): void {
        const sessionKey = request.headers
            ? request.headers[CommonHeaders.X_YA_SESSION_KEY]
            : undefined;

        this.logger.log(
            this.prepareMessage(request, 'request success', response.status),
            {
                method: request.method,
                url: request.url,
                query: request.params,
                status: response.status,
                sessionKey,
                duration,
            },
        );
    }

    private logRequestFailure(
        request: IRequestConfig,
        response: IResponse<any> | undefined,
        error: AxiosError<IResponse<any>> | Error,
        duration: number,
    ): void {
        const sessionKey = request.headers
            ? request.headers[CommonHeaders.X_YA_SESSION_KEY]
            : undefined;

        // DEFAULT CODE FOR UNKNOWN ERRORS
        let status = DEFAULT_UNKNOWN_AXIOS_ERROR_CODE;

        if (response && response.status) {
            status = response.status;
        } else if (
            this.isAxiosError(error) &&
            error.response &&
            error.response.status
        ) {
            status = error.response.status;
        } else if (error instanceof Error) {
            // CODE FOR EXEPTIONS
            status = DEFAULT_EXCPETION_ERROR_CODE;
        }

        this.logger.logError(
            this.prepareMessage(request, 'request failure', status),
            error,
            {
                method: request.method,
                url: request.url,
                query: request.params,
                code: response && response.data && response.data.code,
                status,
                sessionKey,
                duration,
            },
        );
    }

    private isAxiosError(
        error: AxiosError<IResponse<any>> | Error,
    ): error is AxiosError<IResponse<any>> {
        return 'isAxiosError' in error && error.isAxiosError;
    }

    private sendTelemetry(
        request: IRequestConfig,
        response: IResponse<any>,
        duration: number,
    ): void {
        if (this.sendClickHouseStats) {
            this.sendClickHouseStats({
                service: this.getClassName(),
                responseStatus: response.status,
                requestId: this.requestId,
                requestTime: duration,
                requestMethod: request.method as string,
                requestUrl: request.url as string,
                timestamp: new Date().getTime(),
            });
        }
    }

    private sendFaultTelemetry(
        request: IRequestConfig,
        error: IErrorResponse,
        duration: number,
    ): void {
        if (this.sendClickHouseStats) {
            this.sendClickHouseStats({
                service: this.getClassName(),
                responseStatus: error.response ? error.response.status : 999,
                requestId: this.requestId,
                requestTime: duration,
                requestMethod: request.method as string,
                requestUrl: request.url as string,
                timestamp: new Date().getTime(),
            });
        }
    }

    private startTrace(
        request: IRequestConfig,
        spanName?: string,
    ): Span | undefined {
        if (!this.rootSpan) {
            return undefined;
        }

        const headers = {...request.headers};
        const tracer = this.rootSpan.tracer();
        const operationName = `${API_SERVICE_NAME} - (${
            spanName || 'unknown api operation'
        })`;
        const span = tracer.startSpan(operationName, {
            childOf: this.rootSpan,
            tags: {
                method: request.method,
                path: request.url,
                query: request.params,
            },
        });

        tracer.inject(span, FORMAT_HTTP_HEADERS, headers);
        request.headers = headers;

        return span;
    }

    private getSpanName(): string | undefined {
        if (process.env.DISABLE_JAEGER_SPAN_NAME) {
            return undefined;
        }

        try {
            const stackFrame = StackTrace.getSync()?.[3];

            return stackFrame ? stackFrame.functionName : undefined;
        } catch (err) {
            this.logger.log('StackTrace error');

            return undefined;
        }
    }

    private createRequest(
        fullPath: string,
        method: string,
        body: any,
        options?: IRequestConfig,
    ): {
        url: string;
        method: string;
        timeout: number;
        data: any;
    } & IRequestConfig {
        return Object.assign(
            {
                url: fullPath,
                method,
                timeout: this.timeout,
                data: body,
            },
            this.srcParams
                ? {
                      ...options,
                      params: {
                          // затираем srcParams на случай если параметры из урла прокидывались напрямую в api
                          ...omit(options?.params, ['srcParams']),
                          ...this.srcParams,
                      },
                  }
                : options || {},
        );
    }

    private getFullPath(
        relativePath: string,
        options?: IRequestConfig,
    ): string {
        let path = relativePath;
        let baseURL = (options && options.baseURL) || this.baseURL;

        if (!baseURL.endsWith('/')) {
            baseURL = baseURL.concat('/');
        }

        if (path.startsWith('/')) {
            path = path.slice(1);
        }

        return baseURL.concat(path);
    }
}
