import {HTTP_BAD_REQUEST_CODE} from 'server/constants/httpCodes';

import {
    Request,
    Response,
    CoreRequestHandler,
} from '@yandex-data-ui/core/lib/types';

interface IValidateParamsSimple {
    name: string;
    type:
        | EValidateType.DIGIT
        | EValidateType.CHAR
        | EValidateType.CHAR_AND_DIGIT;
    data?: never;
}

interface IValidateParamsEnum {
    name: string;
    type: EValidateType.ENUM;
    data: object;
}

interface IValidateParamsCustom {
    name: string;
    type: EValidateType.CUSTOM;
    data: (value: any) => boolean;
}

type TValidateParams =
    | IValidateParamsSimple
    | IValidateParamsEnum
    | IValidateParamsCustom;

export enum EValidateType {
    'DIGIT' = 'DIGIT',
    'CHAR_AND_DIGIT' = 'CHAR_AND_DIGIT',
    'CHAR' = 'CHAR',
    'ENUM' = 'ENUM',
    'CUSTOM' = 'CUSTOM',
}

const DIGIT_ERROR_REG = /[^0-9]/g;
const CHAR_ERROR_REG = /[^a-zA-Zа-яА-ЯёЁ_-]/g;
const CHAR_AND_DIGIT_ERROR_REG = /[^a-zA-Zа-яА-ЯёЁ0-9_-]/g;

export function validateParams(paramsData: TValidateParams[]) {
    return function (
        target: any,
        propertyKey: string | symbol,
        descriptor: TypedPropertyDescriptor<CoreRequestHandler>,
    ): void {
        const originalFn = descriptor.value;

        if (typeof originalFn !== 'function') {
            throw new TypeError('requiredParams can decorate only functions');
        }

        descriptor.value = function (
            req: Request,
            res: Response,
        ): void | Promise<void> {
            const {query, body} = req;

            for (let i = 0, ln = paramsData.length; i < ln; i++) {
                const param = paramsData[i];
                let requestParam = '';

                if (Object.prototype.hasOwnProperty.call(query, param.name)) {
                    requestParam = query[param.name];
                } else if (
                    Object.prototype.hasOwnProperty.call(body, param.name)
                ) {
                    requestParam = body[param.name];
                }

                if (requestParam !== '') {
                    let value = false;

                    switch (param.type) {
                        case EValidateType.DIGIT:
                            value = !DIGIT_ERROR_REG.test(requestParam);

                            break;
                        case EValidateType.CHAR:
                            value = !CHAR_ERROR_REG.test(requestParam);

                            break;
                        case EValidateType.CHAR_AND_DIGIT:
                            value =
                                !CHAR_AND_DIGIT_ERROR_REG.test(requestParam);

                            break;
                        case EValidateType.ENUM:
                            value = Object.values(param.data).includes(
                                requestParam,
                            );

                            break;
                        case EValidateType.CUSTOM:
                            value = param.data(requestParam);
                    }

                    if (!value) {
                        res.status(HTTP_BAD_REQUEST_CODE).send({
                            statusCode: 'Bad request',
                            fieldError: param.name,
                        });

                        return;
                    }
                }
            }

            return originalFn.call(this, req, res);
        };
    };
}
