/**
 * @typedef {import('@yandex-data-ui/core/lib/types').Request} Request
 * @typedef {import('@yandex-data-ui/core/lib/types').Response} Response
 * @typedef {import('@yandex-data-ui/core/lib/types').NextFunction} NextFunction
 *
 * @typedef {import('../../loggers/avia/types/IAviaVisitSourceIds').IAviaVisitSourceIds} IAviaVisitSourceIds
 */

const {EAviaQueryMarks} = require('./constants/queryMarks');
const {EAviaCookieMarks} = require('./constants/cookieMarks');

const parseUtm = require('./utils/parseUtm');
const validateClid = require('./utils/validateClid');
const getOrganicSource = require('./utils/getOrganicSource');
const getCookieParams = require('./utils/getCookieParams');
const toggleCookie = require('./utils/toggleCookie');
const prepareValue = require('./utils/prepareValue');
const parseUtmsFromCookie = require('./utils/parseUtmsFromCookie');
const shouldFinishUtmSession = require('./utils/shouldFinishUtmSession');
const logInvalidQueryTypes = require('./utils/logInvalidQueryTypes');
const {
    getWizardFlagsFromQuery,
} = require('../../../src/utilities/url/wizard/getWizardFlagsFromQuery');
const {
    getWizardReqIdFromQuery,
} = require('../../../src/utilities/url/wizard/getWizardReqIdFromQuery');

/**
 * Комбинирует метки из параметров запроса и кук, используется в мидлваре.
 *
 * @param {Request} req
 * @param {Response} res
 * @returns {undefined}
 */
function register(req, res) {
    logInvalidQueryTypes(req);

    const {query, cookies} = req;
    const queryUtms = parseUtm(query);
    const hasQueryUtms = Object.keys(queryUtms).length > 0;

    let hasFakeQueryUtmsFor30d = false;

    const queryClid = validateClid(query[EAviaQueryMarks.CLID]),
        queryStid = query[EAviaQueryMarks.STID],
        queryWizardFlags = getWizardFlagsFromQuery(query),
        querySerpUuid = query[EAviaQueryMarks.SERP_UUID],
        queryWizardReqId = getWizardReqIdFromQuery(query);

    let cookieWizardReqId = cookies[EAviaCookieMarks.WIZARD_REQ_ID],
        cookieWizardReqId24 = cookies[EAviaCookieMarks.WIZARD_REQ_ID24],
        cookieWizardFlags = cookies[EAviaCookieMarks.WIZARD_FLAGS],
        cookieSerpUuid = cookies[EAviaCookieMarks.SERP_UUID],
        cookiesUtmSession = cookies[EAviaCookieMarks.UTM_SESSION],
        utms = cookies[EAviaCookieMarks.UTMS],
        cookieClid = cookies[EAviaCookieMarks.CLID],
        cookieStid = cookies[EAviaCookieMarks.STID];

    /*
     * В cookies.utms лежит IAviaUtmMap в json.
     * Если значение сломали, то затираем эту ыкуку.
     */
    const cookieUtms = parseUtmsFromCookie(res, utms, EAviaCookieMarks.UTMS);

    const currentUtms = hasQueryUtms ? queryUtms : cookieUtms;

    /*
     * Когда пользователь пришел без меток, мапим реферрер на фиктивные метки
     * и дальше обрабатываем так, будто метки есть (если реферрер замапился)
     */
    const referrerSource = getOrganicSource(req.get('referrer'));

    if (referrerSource) {
        if (Object.keys(currentUtms).length === 0) {
            hasFakeQueryUtmsFor30d = true;
            queryUtms.utm_source = referrerSource;
        }
    }

    if (hasQueryUtms || hasFakeQueryUtmsFor30d) {
        res.aviaCookie(
            EAviaCookieMarks.UTMS,
            JSON.stringify(queryUtms),
            getCookieParams(30),
        );

        toggleCookie(res, queryClid, cookieClid, EAviaCookieMarks.CLID, 30);
        toggleCookie(res, queryStid, cookieStid, EAviaCookieMarks.STID, 30);

        // если в куках есть utmSession, то нужно сбросить ее, т.к. она появляется после целевого действия,
        // а здесь мы получили новые метки и utmSession будет неактуалной
        if (cookiesUtmSession) {
            res.aviaClearCookie(EAviaCookieMarks.UTM_SESSION);
            delete cookies[EAviaCookieMarks.UTM_SESSION];
        }

        // ютм метки из запроса приоритетней метки колдуна в куках
        if (cookieWizardReqId24) {
            res.aviaClearCookie(EAviaCookieMarks.WIZARD_REQ_ID24);
            cookieWizardReqId24 = null;
        }

        cookieClid = cookieStid = cookiesUtmSession = null;
    }

    if (shouldFinishUtmSession(cookiesUtmSession)) {
        if (cookieUtms) {
            res.aviaClearCookie(EAviaCookieMarks.UTMS);
        }

        if (cookiesUtmSession) {
            res.aviaClearCookie(EAviaCookieMarks.UTM_SESSION);
        }

        cookieClid = null;
        cookieStid = null;
        cookiesUtmSession = null;
    }

    if (queryWizardFlags) {
        res.aviaCookie(
            EAviaCookieMarks.WIZARD_FLAGS,
            queryWizardFlags,
            getCookieParams(1),
        );
        cookieWizardFlags = queryWizardFlags;
    }

    if (queryWizardReqId) {
        res.aviaCookie(
            EAviaCookieMarks.WIZARD_REQ_ID24,
            '1',
            getCookieParams(1),
        );
        cookieWizardReqId24 = '1';

        res.aviaCookie(
            EAviaCookieMarks.WIZARD_REQ_ID,
            queryWizardReqId,
            getCookieParams(1),
        );
        cookieWizardReqId = queryWizardReqId;
    }

    if (querySerpUuid) {
        res.aviaCookie(
            EAviaCookieMarks.SERP_UUID,
            querySerpUuid,
            getCookieParams(1),
        );
        cookieSerpUuid = querySerpUuid;
    }

    utms = parseUtm(currentUtms);

    const clidValue = queryClid || cookieClid;
    const stidValue = queryStid || cookieStid;

    /** @type {IAviaVisitSourceIds} */
    const aviaVisitSourceIds = {
        wizard_flags: prepareValue(cookieWizardFlags),
        wizardReqId: prepareValue(cookieWizardReqId),
        wizardReqId24: Boolean(cookieWizardReqId24),
        serp_uuid: prepareValue(cookieSerpUuid),

        ...utms,
        clid: prepareValue(clidValue),
        stid: prepareValue(stidValue),
    };

    req.aviaVisitSourceIds = aviaVisitSourceIds;
}

/**
 * Чистит метки, используется в целевом действии.
 *
 * @param {Request} req
 * @param {Response} res
 * @returns {undefined}
 */
function track(req, res) {
    const cookieUtmSession = req.cookies[EAviaCookieMarks.UTM_SESSION];
    const cookieUtms = req.cookies[EAviaCookieMarks.UTMS];

    const clearCookie = mark => {
        if (req.cookies[mark] || res.hasCookie(mark)) {
            res.aviaClearCookie(mark);
        }
    };

    if (!cookieUtmSession && cookieUtms) {
        // Метки, для которых нужно сбросить expire (aka жизнь до закрытия браузера)
        const cookiesToUpdate = [
            EAviaCookieMarks.UTMS,
            EAviaCookieMarks.CLID,
            EAviaCookieMarks.STID,
        ];

        const params = {
            httpOnly: true,
        };

        cookiesToUpdate
            .filter(name => Boolean(req.cookies[name]))
            .forEach(name => res.aviaCookie(name, req.cookies[name], params));

        // Эта кука хранит время, которое нужно хранить метки в cookies.utms. Это время проверяется в
        // функции track
        res.aviaCookie(EAviaCookieMarks.UTM_SESSION, Date.now(), params);
    }

    const marksToClear = [
        EAviaCookieMarks.WIZARD_FLAGS,
        EAviaCookieMarks.WIZARD_REQ_ID,
        EAviaCookieMarks.SERP_UUID,
    ];

    marksToClear.forEach(clearCookie);
}

module.exports = {
    register,
    track,
};
