import 'core-js/stable';
import 'regenerator-runtime/runtime';

import 'moment-timezone';
import React from 'react';
import ReactDOM from 'react-dom';
import {Provider} from 'react-redux';
import {Router} from 'react-router-dom';
import {loadableReady} from '@loadable/component';
import polyfillsPromise from 'polyfills/index';

import {ECommonGoal} from 'utilities/metrika/types/goals/common';

import {StoreInterface} from 'reducers/storeTypes';
import createReduxStore from 'reducers/createClientReduxStore';

import {setDateLocale} from 'utilities/dateUtils';
import locales from 'utilities/dateUtils/locales';
import {createGlobalTokenGetter} from 'utilities/csrfToken/createGlobalTokenGetter';
import {hit, reachGoal} from 'utilities/metrika';
import browserHistory from 'utilities/browserHistory/browserHistory';
import scrollTo from 'utilities/dom/scrollTo';
import checkDocumentOnLoad from './helpers/checkDocumentOnLoad';
import appData from './utilities/appData/appData';
import {csrfTokenIsOutOfDate} from 'utilities/csrfToken/csrfTokenIsOutOfDate';
import {storeSrcParams} from 'utilities/srcParams/srcParams';
import pageManager from 'utilities/pageManager/pageManager';
import finalizeRumScores from 'utilities/rum/finalizeRumScores';
import {syncAuth} from 'utilities/userInfo/syncAuth';
import {isAuthUser} from 'utilities/userInfo/isAuthUser';
import {reloadPage} from 'utilities/browser/reloadPage';
import {logDebug} from 'utilities/logger/logDebug';

import App from 'containers/App/App';

const reduxStore = createReduxStore();
const initialStoreState: StoreInterface = reduxStore.getState();
const {
    common: {
        userInfo,
        goals: initialGoals,
        environmentConfig: {csrfTokenValidTill},
    },
} = initialStoreState;
const rootNode = document.getElementById('app');

initialGoals.forEach(goal => reachGoal(goal));
createGlobalTokenGetter(initialStoreState);
storeSrcParams();
syncAuth(isAuthUser(userInfo));

setDateLocale(locales.RU);

if (window.appData) {
    appData.data = window.appData;
    delete window.appData;
}

if (browserHistory) {
    let prevModalId: string | undefined;

    browserHistory.listen((location, action) => {
        if (csrfTokenIsOutOfDate(csrfTokenValidTill)) {
            reachGoal(ECommonGoal.PAGE_RELOAD_CSRF_TOKEN_OUTDATED);
            logDebug({message: 'Page reload due CSRF token is out of date'});

            // ждем немного для завершения асинхронных вызовов которые не возвращают Promise
            setTimeout(reloadPage, 300);

            return;
        }

        const {pathname, search, state, hash} = location;

        /**
         * Не делаем hit при открытии модала,
         * Не делаем hit при нажатии кнопки назад, когда предыдущее действие было открытием модала
         */
        if (
            (action === 'PUSH' && !state?.modalId) ||
            (action === 'POP' && !prevModalId)
        ) {
            hit(`${pathname}${search}`);
        }

        prevModalId = state?.modalId;

        finalizeRumScores();

        if (typeof state?.scrollTop === 'number') {
            scrollTo({top: state.scrollTop});

            return;
        }

        // На время, пока для авиа фильтров испульзуем фрагмент, конфликтует с якорями,
        // договорились перенести авиа-фильтры из хеша в кверю
        const isAviaFiltersHash = /^#empty$|=/.test(hash);

        if (
            (!hash || isAviaFiltersHash) &&
            action === 'PUSH' &&
            state?.scrollTop === undefined
        ) {
            /**
             * Для всех переходов с добавлением нового состояния в историю (PUSH)
             * и когда location не содержит хеш
             * и когда не был передан state.scrollTop
             * всегда проскроливаем страницу к верху
             */
            scrollTo({top: 0});
        }
    });
}

(async (): Promise<void> => {
    await pageManager.init({
        isProductionEnv: appData.isProductionEnv,
        pageToken: initialStoreState.common.csrfToken.value ?? null,
    });

    await checkDocumentOnLoad();
    await polyfillsPromise;
    await loadableReady();

    ReactDOM.hydrate(
        <React.StrictMode>
            <Provider store={reduxStore}>
                <Router
                    // @ts-ignore
                    history={browserHistory}
                >
                    <App />
                </Router>
            </Provider>
        </React.StrictMode>,
        rootNode,
    );
})();
