import React, {MouseEvent, RefObject, useCallback} from 'react';

import IActionItem from './types/IActionItem';
import EPopupDirection from 'components/Popup/types/EPopupDirection';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {delay} from 'utilities/async/delay';

import MobileActionsDialog from './components/MobileActionsDialog/MobileActionsDialog';
import DesktopActionsDialog from './components/DesktopActionsDialog/DesktopActionsDialog';
import {TIconSize} from 'components/TextWithIcon/TextWithIcon';

export interface IActionsDialogProps
    extends IWithQaAttributes,
        IWithQaAttributes {
    actions: IActionItem[];
    isOpened: boolean;
    buttonRef: RefObject<HTMLElement>;
    popupDirections?: EPopupDirection[];
    popupSecondaryOffset?: number;
    iconSize?: TIconSize;
    /**
     * Задержка закрытия после выбора элемента
     */
    delayAfterSelect?: number;
    onClose(): void;
}

const ActionsDialog: React.FC<IActionsDialogProps> = props => {
    const {
        actions,
        isOpened,
        onClose,
        buttonRef,
        popupDirections,
        popupSecondaryOffset,
        iconSize,
        delayAfterSelect,
    } = props;
    const deviceType = useDeviceType();

    const handleClick = useCallback(
        async (event: MouseEvent, action: IActionItem) => {
            try {
                await action.onClick(event);
            } catch (e) {
                console.error(e);

                return;
            }

            if (delayAfterSelect) {
                await delay(delayAfterSelect);
            }

            onClose();
        },
        [delayAfterSelect, onClose],
    );

    if (deviceType.isMobile) {
        return (
            <MobileActionsDialog
                actions={actions}
                isOpened={isOpened}
                iconSize={iconSize}
                onItemClick={handleClick}
                onClose={onClose}
                {...prepareQaAttributes(props)}
            />
        );
    }

    return (
        <DesktopActionsDialog
            actions={actions}
            isOpened={isOpened}
            buttonRef={buttonRef}
            iconSize={iconSize}
            directions={popupDirections}
            secondaryOffset={popupSecondaryOffset}
            onItemClick={handleClick}
            onClose={onClose}
            {...prepareQaAttributes(props)}
        />
    );
};

export default React.memo(ActionsDialog);
