import React, {MouseEvent, useCallback, useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';
import IActionItem from 'components/ActionsDialog/types/IActionItem';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';

import TextWithIcon, {TIconSize} from 'components/TextWithIcon/TextWithIcon';
import SpinnerAsIcon from 'components/ActionsDialog/components/ActionItem/components/SpinnerAsIcon/SpinnerAsIcon';

import cx from './ActionItem.scss';

interface IActionItemProps extends IWithClassName, IWithQaAttributes {
    item: IActionItem;
    iconSize?: TIconSize;
    onClick(e: MouseEvent, item: IActionItem): void;
}

const ActionItem: React.FC<IActionItemProps> = props => {
    const {className, item, iconSize = 16, onClick} = props;

    const deviceType = useDeviceType();

    const handleClick: React.MouseEventHandler = useCallback(
        e => {
            onClick(e, item);
        },
        [item, onClick],
    );

    const text = useMemo(() => {
        return (
            <div className={cx('content')}>
                <span>{item.title}</span>
                {Boolean(item.description) && <span>{item.description}</span>}
            </div>
        );
    }, [item.description, item.title]);

    return (
        <button
            className={cx(className, 'root', deviceMods('root', deviceType), {
                root_loading: item.loading,
            })}
            key={item.id}
            disabled={item.disabled}
            onClick={handleClick}
            {...prepareQaAttributes(props)}
        >
            <TextWithIcon
                textClassName={cx('text')}
                text={text}
                size="m"
                spaceBetween={3}
                iconSize={iconSize}
                iconLeft={item.loading ? SpinnerAsIcon : item.icon}
                iconLeftClassName={cx('icon')}
            />
        </button>
    );
};

export default ActionItem;
