import {useEffect, FC, memo} from 'react';
import {useSelector} from 'react-redux';
import {IntersectionOptions, useInView} from 'react-intersection-observer';

import {EAdFoxBannerType, EAdFoxBannerPosition} from 'types/AdFox';
import {IWithClassName} from 'types/withClassName';

import {adFoxParamsSelector} from 'selectors/common/banners';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import useCashedAdBlockIsEnabled from 'utilities/hooks/useCashedAdBlockIsEnabled';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {useNeedToLoadDynamicResource} from 'utilities/hooks/useNeedToLoadDynamicResource';

import AdsScript from 'components/AdsScript/AdsScript';

import cx from './AdFoxBanner.scss';

const INTERSECTION_OBSERVER_OPTIONS: IntersectionOptions = {
    rootMargin: '500px',
    triggerOnce: true,
};

interface IAdFoxBannerProps extends IWithClassName {
    fixed?: boolean;
    lazyLoad?: boolean;
    lazyLoadOptions?: IntersectionObserverInit;
    type: EAdFoxBannerType;
    position: EAdFoxBannerPosition;
    // @see https://yandex.ru/support/adfox-sites/codes/code.html#code-loader-expansion__callback
    onLoad?: (data: {bundleName: string; bundleParams: object}) => void;
    onRender?: (data: {bundleName: string; bundleParams: object}) => void;
}

const DECODED_TYPES = {
    [EAdFoxBannerType.Inline]: 'inline',
    [EAdFoxBannerType.Tile]: 'tile',
    [EAdFoxBannerType.Popup]: 'popup',
    [EAdFoxBannerType.BottomLine]: 'bottomLine',
};

const AdFoxBanner: FC<IAdFoxBannerProps> = ({
    type,
    fixed,
    position,
    className,
    lazyLoad,
    onRender,
    onLoad,
    lazyLoadOptions,
}) => {
    const containerId = `${type}${position}`;
    const adFoxParams = useSelector(adFoxParamsSelector);

    const adBlockIsEnabled = useCashedAdBlockIsEnabled();

    const {value: needToLoadScriptByScroll, setTrue: loadScriptByScroll} =
        useBoolean(!lazyLoad);

    const needToLoadScriptByTimeout = useNeedToLoadDynamicResource({
        timeoutProportion: 10,
        maxProportionTimeout: 15000,
    });

    const [ref, hasIntersected] = useInView({
        skip: !lazyLoad,
        ...(lazyLoadOptions || INTERSECTION_OBSERVER_OPTIONS),
    } as any);

    useEffect(() => {
        if (!adFoxParams) {
            return;
        }

        const createCallback = (): void => {
            window.Ya.adfoxCode.create({
                ownerId: adFoxParams.ownerId,
                containerId,
                params: {
                    pp: position,
                    ps: adFoxParams.siteId,
                    p2: type,
                },
                onLoad,
                onRender,
            });
        };

        (window.yaContextCb ||= []).push(createCallback);

        /**
         * Удаляем из списка для случая, когда рекламный скрипт еще не прогрузился,
         * а мы покидаем страницу. Т.к. на новой странице может быть такой же элемент,
         * куда вставляется баннер и тогда отобразится 2 баннера одновременно.
         */
        return (): void => {
            if (!Array.isArray(window.yaContextCb)) {
                return;
            }

            const bannerIndex = window.yaContextCb.findIndex(
                localCallback => localCallback === createCallback,
            );

            if (bannerIndex !== -1) {
                window.yaContextCb = [
                    ...window.yaContextCb.slice(0, bannerIndex),
                    ...window.yaContextCb.slice(bannerIndex + 1),
                ];
            }
        };
    }, [position, type, containerId, adFoxParams, onLoad, onRender]);

    useEffect(() => {
        if (lazyLoad && hasIntersected) {
            loadScriptByScroll();
        }
    }, [hasIntersected, lazyLoad, loadScriptByScroll]);

    if (!adFoxParams) {
        return null;
    }

    return (
        <>
            {(needToLoadScriptByScroll || needToLoadScriptByTimeout) && (
                <AdsScript />
            )}

            <div
                ref={ref}
                id={containerId}
                className={cx(
                    'root',
                    `root_type_${DECODED_TYPES[type]}`,
                    fixed && !adBlockIsEnabled && 'root_fixed',
                    className,
                )}
                {...prepareQaAttributes('ad-fox-banner')}
            />
        </>
    );
};

export default memo(AdFoxBanner);
