import React, {
    FunctionComponent,
    ComponentType,
    ReactNode,
    useCallback,
    memo,
    useMemo,
    CSSProperties,
} from 'react';

import {IWithClassName} from 'types/withClassName';
import IIconAdvantageProps from 'components/Advantages/types/IIconAdvantageProps';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {
    getQa,
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import StaticContent from 'containers/StaticContent/StaticContent';

import Swipeable, {
    ISwipeableRenderItemParams,
} from 'components/Swipeable/Swipeable';
import Advantage from 'components/Advantages/components/Advantage/Advantage';

import cx from './Advantages.scss';

export type TAdvantage = {
    title?: string;
    description: ReactNode;
    IconComponent: ComponentType<IIconAdvantageProps>;
    iconWidth: number;
    iconHeight: number;
};

export type TSwipeableRenderItemParams = Omit<
    ISwipeableRenderItemParams<TAdvantage>,
    'isCurrent'
>;

interface IAdvantagesProps extends IWithClassName, IWithQaAttributes {
    title?: ReactNode;
    titleMaxWidth?: CSSProperties['maxWidth'];
    advantages: TAdvantage[];
}

const Advantages: FunctionComponent<IAdvantagesProps> = props => {
    const {className, title, titleMaxWidth, advantages} = props;

    const deviceType = useDeviceType();
    const rootQa = getQa(props);

    const renderer = useCallback(
        ({data, index}: TSwipeableRenderItemParams): ReactNode => (
            <Advantage
                key={index}
                className={cx('item')}
                title={data.title}
                description={data.description}
                IconComponent={data.IconComponent}
                iconWidth={data.iconWidth}
                iconHeight={data.iconHeight}
                descriptionTextSize="l"
                {...prepareQaAttributes({
                    parent: rootQa,
                    current: 'advantage',
                    key: index,
                })}
            />
        ),
        [rootQa],
    );

    const content = useMemo(() => {
        return (
            <>
                {title && (
                    <div
                        className={cx('title')}
                        style={{maxWidth: titleMaxWidth}}
                        {...prepareQaAttributes({
                            parent: rootQa,
                            current: 'title',
                        })}
                    >
                        {title}
                    </div>
                )}
                <div className={cx('advantages')}>
                    {deviceType.isMobile ? (
                        <Swipeable
                            className={cx('advantagesSwipeable')}
                            sideThreshold={0.1}
                            renderItem={renderer}
                            itemsData={advantages}
                            itemVerticalAlignment="start"
                            showNav
                        />
                    ) : (
                        <>
                            {advantages.map((data, index) =>
                                renderer({data, index}),
                            )}
                        </>
                    )}
                </div>
            </>
        );
    }, [
        advantages,
        deviceType.isMobile,
        renderer,
        rootQa,
        title,
        titleMaxWidth,
    ]);

    const rootClassName = cx('root', deviceMods('root', deviceType), className);

    // Используем обёртку, т.к. в мобильной версии используется Swipeable
    const Wrapper = deviceType.isMobile ? React.Fragment : StaticContent;

    return (
        <Wrapper>
            <div className={rootClassName} {...prepareQaAttributes(rootQa)}>
                {content}
            </div>
        </Wrapper>
    );
};

export default memo(Advantages);
