import {FC, useMemo, createElement, memo} from 'react';

import {IWithClassName} from 'types/withClassName';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import ExcludeIcon from 'icons/16/Exclude';
import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import BaggageIcon from 'components/BaggageIconWithWeight/components/BaggageIcon/BaggageIcon';
import CarryOnIcon from 'components/BaggageIconWithWeight/components/CarryOnIcon/CarryOnIcon';

import cx from './BaggageIconWithWeight.scss';

export enum EBaggageWithWeightIconType {
    BAGGAGE = 'baggage',
    CARRY_ON = 'carryOn',
}

export enum EBaggageWithWeightIconSize {
    S = 's',
    M = 'm',
}

interface IBaggageIconWithWeightProps
    extends IWithClassName,
        IWithQaAttributes {
    type: EBaggageWithWeightIconType;
    included: boolean;
    weight?: number;
    count?: number;
    allowRuble?: boolean;
    currencyPrefix?: string;
}

const BaggageIconWithWeight: FC<IBaggageIconWithWeightProps> = props => {
    const {
        className,
        type,
        weight,
        included,
        count,
        allowRuble = false,
        currencyPrefix = '',
    } = props;

    const content = useMemo(() => {
        if (!included || weight) {
            const size = 11;

            const contentClassName =
                type === EBaggageWithWeightIconType.BAGGAGE
                    ? 'baggageIconContent'
                    : 'carryOnIconContent';

            return (
                <div className={cx('content', contentClassName)}>
                    {weight ?? <ExcludeIcon width={size} height={size} />}
                </div>
            );
        }

        return null;
    }, [included, type, weight]);

    const icon = useMemo(
        () =>
            createElement(
                type === EBaggageWithWeightIconType.BAGGAGE
                    ? BaggageIcon
                    : CarryOnIcon,
                {
                    size: EBaggageWithWeightIconSize.M,
                    empty: Boolean(!included || weight),
                },
            ),
        [included, type, weight],
    );

    return (
        <Flex
            className={cx(
                'root',
                included ? 'root_included' : 'root_notIncluded',
                className,
            )}
            inline
            nowrap
            {...prepareQaAttributes(props)}
        >
            <div className={cx('iconWrapper')}>
                {icon}
                {content}
            </div>

            {count && count > 1 ? (
                <Text className={cx('count')} size="m" color="current">
                    x{count}
                </Text>
            ) : null}
            {!included && allowRuble && (
                <Text className={cx('rubble')} size="m">
                    {`${currencyPrefix}₽`}
                </Text>
            )}
        </Flex>
    );
};

export default memo(BaggageIconWithWeight);
