import {
    ReactNode,
    useCallback,
    useContext,
    useEffect,
    useMemo,
    useState,
} from 'react';
import {FieldMetaState, useForm, useFormState} from 'react-final-form';
import _get from 'lodash/get';

import {EDocumentType} from 'constants/document/documentTypes';
import {RU_CITIZENSHIP_CODE2} from 'constants/document/citizenship';
import {EFieldName} from './constants/fieldNames';
import {
    DOCUMENT_TYPES,
    DOCUMENTS_WITH_CITIZEN,
    DOCUMENTS_WITH_DATE,
} from './constants/documentTypes';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {ICountry} from 'types/common/ICountry';
import {TDocumentType} from 'types/common/document/TDocumentType';
import {IPassengerWithDocumentsDTO} from 'server/api/TravelersApi/types/IPassengerDTO';
import {IFormValidationInfo} from 'types/common/validation/form';
import {
    IAutoCompleteDocument,
    IFormDocument,
    ISuggestDocument,
} from './types/IDocument';
import {IFieldsValidation} from './types/IFieldsValidation';

import {deviceMods} from 'utilities/stylesUtils';
import {getInitialDocumentType} from './utilities/getInitialDocumentType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import setFormValues from './utilities/setFormValues';

import * as i18nBlock from 'i18n/components-BookingPassengerForm';

import Form from 'components/Form/Form';
import Text from 'components/Text/Text';
import Select from 'components/Select/Select';
import FormField from 'components/FormField/FormField';
import RadioButton from 'components/RadioButton/RadioButton';
import {TControlRenderFunc} from 'components/FormField/components/Field/Field';
import DateInput from './components/DateInput/DateInput';
import SuggestInput from './components/SuggestInput/SuggestInput';
import CountrySelect from './components/CountrySelect/CountrySelect';
import PatronymicInput from './components/PatronymicInput/PatronymicInput';
import DocumentTypeField from './components/DocumentTypeField/DocumentTypeField';
import DocumentNumberInput from './components/DocumentNumberInput/DocumentNumberInput';
import PassengerIntentsList from './components/PassengerIntentsList/PassengerIntentsList';
import {DocumentsContextProvider} from './components/DocumentsContext/DocumentsContext';

import cx from './BookingPassengerForm.scss';

export type {IFormDocument};

export interface IBookingPassengerFormProps<TAllValues extends object = object>
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    title?: ReactNode;
    message?: ReactNode;
    countries?: ICountry[];
    availableDocuments?: TDocumentType[];
    hasPatronymic?: boolean;
    hasDocumentDate?: boolean;
    isOnlyName?: boolean;
    isLatinName?: boolean;
    /** Флаг, при котором disabled поле гражданства не отображается */
    hideDisabledCitizenship?: boolean;
    disabled?: boolean;
    passengers?: IPassengerWithDocumentsDTO[];
    initialValues?: IFormDocument;
    /** Правила валидации для фильтрации по ним списка документов */
    validationInfo?: IFormValidationInfo;
    fieldsValidation?: IFieldsValidation<TAllValues>;
    excludeRuCitizenForOtherDocument?: boolean;
    onChangeError?: (fieldName: string, meta: FieldMetaState<string>) => void;
    handleIntentSelect?: (document: IAutoCompleteDocument) => void;
    handleSuggestSelect?: (document: IAutoCompleteDocument) => void;
    renderSuggestOption?: (document: ISuggestDocument) => ReactNode;
    /**
     * Не очищать ли поля documents от заведомо невалидных значений
     * @default false
     */
    keepInvalidFields?: boolean;
}

export function BookingPassengerForm<TAllValues extends object = object>(
    props: IBookingPassengerFormProps<TAllValues>,
): JSX.Element {
    const {
        className,
        deviceType,
        hasPatronymic,
        hasDocumentDate,
        isLatinName,
        isOnlyName,
        disabled,
        passengers,
        initialValues,
        validationInfo,
        availableDocuments,
        fieldsValidation,
        excludeRuCitizenForOtherDocument,
        hideDisabledCitizenship,
        message,
        onChangeError,
        handleIntentSelect,
        handleSuggestSelect,
        renderSuggestOption,
        keepInvalidFields = false,
        ...rest
    } = props;

    const formApi = useForm();
    const formGroupId = useContext(Form.FieldGroupContext);

    const [documentType, setDocumentType] = useState<TDocumentType>(
        getInitialDocumentType(initialValues?.documentType, availableDocuments),
    );

    const {initialValues: stateInitialValues} = useFormState({
        subscription: {initialValues: true},
    });

    useEffect(() => {
        const initialDocumentType: EDocumentType | undefined = _get(
            stateInitialValues,
            formGroupId,
        )?.[EFieldName.documentType];

        if (initialDocumentType) {
            setDocumentType(initialDocumentType);
        }
    }, [formGroupId, stateInitialValues]);

    const getQaAttribute = useCallback(
        (name: string, key?: string): IWithQaAttributes =>
            prepareQaAttributes({
                current: name.split('.').pop(),
                parent: props,
                key,
            }),
        [props],
    );

    const genderRadioButton = useMemo(() => {
        const renderControl: TControlRenderFunc = (
            {input},
            {error, controlRef},
        ) => {
            return (
                <div ref={controlRef}>
                    <RadioButton
                        {...input}
                        {...getQaAttribute(input.name)}
                        state={error ? RadioButton.ERROR : RadioButton.NORMAL}
                        size="l"
                        width="max"
                        data-name={input.name}
                    >
                        <RadioButton.Radio
                            value="male"
                            controlAttrs={{
                                ...getQaAttribute(input.name, 'male'),
                            }}
                        >
                            {hasPatronymic
                                ? i18nBlock.sexDashValueDashMaleDashFull()
                                : i18nBlock.sexDashValueDashMale()}
                        </RadioButton.Radio>

                        <RadioButton.Radio
                            value="female"
                            controlAttrs={{
                                ...getQaAttribute(input.name, 'female'),
                            }}
                        >
                            {hasPatronymic
                                ? i18nBlock.sexDashValueDashFemaleDashFull()
                                : i18nBlock.sexDashValueDashFemale()}
                        </RadioButton.Radio>
                    </RadioButton>
                </div>
            );
        };

        return (
            <FormField
                className={cx('control', 'fields__sex')}
                name={EFieldName.sex}
                title={i18nBlock.sexDashTitle()}
                deviceType={deviceType}
                control={renderControl}
                disabled={disabled}
                initialValue={initialValues?.sex}
                onChangeError={onChangeError}
            />
        );
    }, [
        deviceType,
        disabled,
        getQaAttribute,
        hasPatronymic,
        initialValues?.sex,
        onChangeError,
    ]);

    const documentTypeComponent = useMemo(() => {
        const documentTypes = availableDocuments
            ? DOCUMENT_TYPES.filter(document =>
                  availableDocuments.includes(document.value),
              )
            : DOCUMENT_TYPES;

        const renderDocumentTypeSelect: TControlRenderFunc = (
            {input},
            {error, controlRef},
        ) => {
            return (
                <Select
                    {...input}
                    {...getQaAttribute(input.name)}
                    options={documentTypes}
                    size="l"
                    width="max"
                    theme="outlined"
                    error={Boolean(error)}
                    innerRef={controlRef}
                />
            );
        };

        return (
            <DocumentTypeField
                className={cx('control', 'fields__documentType')}
                deviceType={deviceType}
                initialDocumentType={documentType}
                disabled={disabled}
                renderDocumentTypeSelect={renderDocumentTypeSelect}
                excludeRuCitizenForOtherDocument={
                    excludeRuCitizenForOtherDocument
                }
                onDocumentTypeChange={setDocumentType}
                onChangeError={onChangeError}
            />
        );
    }, [
        availableDocuments,
        deviceType,
        disabled,
        documentType,
        excludeRuCitizenForOtherDocument,
        getQaAttribute,
        onChangeError,
    ]);

    const documentNumber = useMemo(() => {
        const {mask, placeholder, hint, keyboardInputMode} =
            DOCUMENT_TYPES.find(x => x.value === documentType) || {};

        const parseDocumentNumber = (number: string): string =>
            number.replace('\u2000', '');

        return (
            <FormField
                className={cx('field', 'control', cx('fields__documentNumber'))}
                name={EFieldName.documentNumber}
                title={i18nBlock.documentNumberDashTitle()}
                deviceType={deviceType}
                hint={hint}
                parse={parseDocumentNumber}
                disabled={disabled}
                initialValue={initialValues?.documentNumber}
                validate={fieldsValidation?.documentNumber}
                onChangeError={onChangeError}
                control={(
                    {input},
                    {error, inputRef, controlRef},
                ): ReactNode => (
                    <DocumentNumberInput
                        {...input}
                        {...getQaAttribute(input.name)}
                        inputRef={(currentInputRef): void => {
                            inputRef.current = currentInputRef;
                        }}
                        inputMode={keyboardInputMode}
                        innerRef={controlRef}
                        mask={mask}
                        placeholder={placeholder}
                        state={error ? 'error' : undefined}
                    />
                )}
            />
        );
    }, [
        deviceType,
        disabled,
        documentType,
        fieldsValidation?.documentNumber,
        getQaAttribute,
        initialValues?.documentNumber,
        onChangeError,
    ]);

    const isExcludedRuCitizenship = useMemo(
        () =>
            Boolean(excludeRuCitizenForOtherDocument) &&
            documentType === EDocumentType.OTHER,

        [documentType, excludeRuCitizenForOtherDocument],
    );

    const availableCountries = useMemo(
        () =>
            (rest.countries ?? []).filter(country =>
                isExcludedRuCitizenship
                    ? country.code2 !== RU_CITIZENSHIP_CODE2
                    : Boolean(country.code2),
            ),
        [isExcludedRuCitizenship, rest.countries],
    );

    const countrySelector = useMemo(() => {
        const initialCitizenship =
            initialValues?.citizenship ||
            (isExcludedRuCitizenship ? undefined : RU_CITIZENSHIP_CODE2);

        const renderCountrySelect: TControlRenderFunc = (
            {input},
            {error, controlRef},
        ) => (
            <CountrySelect
                {...input}
                {...getQaAttribute(input.name)}
                countries={availableCountries}
                innerRef={controlRef}
                error={Boolean(error)}
            />
        );

        return (
            <FormField
                className={cx('control', 'fields__citizenship', {
                    fields_hidden:
                        hideDisabledCitizenship &&
                        !DOCUMENTS_WITH_CITIZEN.includes(documentType),
                })}
                name={EFieldName.citizenship}
                title={i18nBlock.citizenshipDashTitle()}
                deviceType={deviceType}
                control={renderCountrySelect}
                disabled={
                    disabled || !DOCUMENTS_WITH_CITIZEN.includes(documentType)
                }
                initialValue={initialCitizenship}
                onChangeError={onChangeError}
            />
        );
    }, [
        initialValues?.citizenship,
        isExcludedRuCitizenship,
        hideDisabledCitizenship,
        documentType,
        deviceType,
        disabled,
        onChangeError,
        getQaAttribute,
        availableCountries,
    ]);

    const fieldsClassModifier = useMemo(() => {
        if (isOnlyName) {
            return 'fields_grid_onlyName';
        }

        if (!isOnlyName && !hasPatronymic && hasDocumentDate) {
            return 'fields_grid_withoutPatronymicWithDate';
        }

        if (!isOnlyName && hasPatronymic && !hasDocumentDate) {
            return 'fields_grid_withPatronymicWithoutDate';
        }

        if (!isOnlyName && !hasPatronymic && !hasDocumentDate) {
            return 'fields_grid_withoutPatronymicWithoutDate';
        }

        return 'fields_grid_withPatronymicWithDate';
    }, [hasDocumentDate, hasPatronymic, isOnlyName]);

    const renderDateInput: TControlRenderFunc = useCallback(
        ({input}, {error, controlRef}) => (
            <DateInput
                {...input}
                {...getQaAttribute(input.name)}
                name={input.name}
                size="l"
                state={error ? 'error' : undefined}
                innerRef={controlRef}
            />
        ),
        [getQaAttribute],
    );

    const updateDocumentValues = useCallback(
        (document: IAutoCompleteDocument) => {
            setFormValues(formApi, formGroupId, document, availableDocuments, {
                hasPatronymic,
                isOnlyName,
            });
        },
        [availableDocuments, formApi, formGroupId, hasPatronymic, isOnlyName],
    );

    const onSuggestSelect = useCallback(
        (document: IAutoCompleteDocument) => {
            (handleSuggestSelect ?? updateDocumentValues)(document);
        },
        [handleSuggestSelect, updateDocumentValues],
    );

    const onIntentSelect = useCallback(
        (document: IAutoCompleteDocument) => {
            (handleIntentSelect ?? updateDocumentValues)(document);
        },
        [handleIntentSelect, updateDocumentValues],
    );

    return (
        <DocumentsContextProvider
            passengers={passengers}
            countries={availableCountries}
            avaliableDocuments={availableDocuments}
            validationInfo={validationInfo}
            excludeRuCitizenForOtherDocument={excludeRuCitizenForOtherDocument}
            isOnlyName={isOnlyName}
            hasPatronymic={hasPatronymic}
            isLatinName={isLatinName}
            keepInvalidFields={keepInvalidFields}
        >
            <fieldset
                className={cx(
                    'root',
                    deviceMods('root', deviceType),
                    className,
                )}
                {...prepareQaAttributes(props)}
            >
                {rest.title && (
                    <Text
                        className={cx('title')}
                        size="l"
                        tag="legend"
                        weight="bold"
                    >
                        {rest.title}
                    </Text>
                )}

                {passengers && (
                    <PassengerIntentsList
                        className={cx('intentsList')}
                        deviceType={deviceType}
                        onIntentSelect={onIntentSelect}
                        {...prepareQaAttributes({
                            parent: rest,
                            current: 'intents',
                        })}
                    />
                )}

                {message}

                <div className={cx('fields', fieldsClassModifier)}>
                    <SuggestInput
                        className={cx('fields__lastName', 'control')}
                        deviceType={deviceType}
                        bookInputProps={{
                            deviceType,
                            name: EFieldName.lastName,
                            title: isLatinName
                                ? i18nBlock.lastNameDashTitleDashLatin()
                                : i18nBlock.lastNameDashTitle(),
                            disabled: disabled,
                            initialValue: initialValues?.lastName,
                            onChangeError,
                            ...prepareQaAttributes(rest),
                        }}
                        onSuggestSelect={onSuggestSelect}
                        renderOption={renderSuggestOption}
                    />

                    <SuggestInput
                        className={cx('fields__firstName', 'control')}
                        deviceType={deviceType}
                        bookInputProps={{
                            deviceType,
                            name: EFieldName.firstName,
                            title: isLatinName
                                ? i18nBlock.firstNameDashTitleDashLatin()
                                : i18nBlock.firstNameDashTitle(),
                            disabled: disabled,
                            initialValue: initialValues?.firstName,
                            onChangeError,
                            ...prepareQaAttributes(rest),
                        }}
                        onSuggestSelect={onSuggestSelect}
                        renderOption={renderSuggestOption}
                    />

                    {hasPatronymic && (
                        <PatronymicInput
                            className={cx('fields__patronymicName', 'control')}
                            suggestClassName={cx('fields__patronymicNameInput')}
                            deviceType={deviceType}
                            bookInputProps={{
                                deviceType,
                                name: EFieldName.patronymicName,
                                title: isLatinName
                                    ? i18nBlock.patronymicNameDashTitleDashLatin()
                                    : i18nBlock.patronymicNameDashTitle(),
                                disabled: disabled,
                                initialValue: initialValues?.patronymicName,
                                onChangeError,
                                ...prepareQaAttributes(rest),
                            }}
                            initialCheckbox={
                                initialValues?.isPatronymicDisabled
                            }
                            onSuggestSelect={onSuggestSelect}
                            renderOption={renderSuggestOption}
                        />
                    )}

                    {!isOnlyName && genderRadioButton}

                    {!isOnlyName && (
                        <FormField
                            className={cx('control', 'fields__birthday')}
                            name={EFieldName.birthdate}
                            title={i18nBlock.birthdayDashTitle()}
                            hint={i18nBlock.birthdayDashHint()}
                            deviceType={deviceType}
                            control={renderDateInput}
                            disabled={disabled}
                            initialValue={initialValues?.birthdate}
                            validate={fieldsValidation?.birthdate}
                            onChangeError={onChangeError}
                        />
                    )}

                    {!isOnlyName && documentTypeComponent}

                    {!isOnlyName && documentNumber}

                    {!isOnlyName && countrySelector}

                    {hasDocumentDate && !isOnlyName && (
                        <FormField
                            className={cx(
                                'control',
                                'fields__documentValidDate',
                            )}
                            name={EFieldName.documentValidDate}
                            title={i18nBlock.documentValidDateDashTitle()}
                            hint={i18nBlock.documentValidDateDashHint()}
                            deviceType={deviceType}
                            disabled={
                                disabled ||
                                !DOCUMENTS_WITH_DATE.includes(documentType)
                            }
                            control={renderDateInput}
                            initialValue={initialValues?.documentValidDate}
                            onChangeError={onChangeError}
                        />
                    )}
                </div>
            </fieldset>
        </DocumentsContextProvider>
    );
}

export default BookingPassengerForm;
