import React, {ReactNode, useCallback, useMemo} from 'react';
import {FieldMetaState} from 'react-final-form';
import {v4 as uuid} from 'uuid';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import Input from 'components/Input/Input';
import FormField from 'components/FormField/FormField';

export interface IBookingInputProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    name: string;
    title: string;
    placeholder?: string;
    disabled?: boolean;
    initialValue?: string;
    isHiddenError?: boolean;
    hint?: string;

    onChange?: (
        event: React.ChangeEvent<HTMLInputElement>,
        value: string,
    ) => void;
    onFocus?: (event?: React.FocusEvent<HTMLElement> | undefined) => void;
    onBlur?: (event?: React.FocusEvent<HTMLElement> | undefined) => void;
    onKeyDown?: (event: React.KeyboardEvent<HTMLInputElement>) => void;
    onChangeError?: (fieldName: string, meta: FieldMetaState<string>) => void;

    inputRef?: (input: HTMLInputElement | null) => void;
}

const BookingInput: React.FC<IBookingInputProps> = ({
    className,
    deviceType,
    name,
    title,
    placeholder,
    disabled,
    isHiddenError,
    initialValue,
    hint,
    onChange,
    onFocus,
    onBlur,
    onKeyDown,
    onChangeError,
    inputRef: propsInputRef,
    ...rest
}) => {
    const randomName = useMemo(uuid, []);

    const handleChange = useCallback(
        (event: React.ChangeEvent<HTMLInputElement>) => {
            onChange?.(event, event.target.value);
        },
        [onChange],
    );

    return (
        <FormField
            className={className}
            id={randomName}
            name={name}
            title={title}
            deviceType={deviceType}
            isHiddenError={isHiddenError}
            onChange={handleChange}
            onFocus={onFocus}
            onBlur={onBlur}
            inputRef={propsInputRef}
            initialValue={initialValue}
            hint={hint}
            onChangeError={onChangeError}
            control={({input}, {error, inputRef, controlRef}): ReactNode => (
                <Input
                    {...input}
                    {...prepareQaAttributes({
                        current: name,
                        parent: rest,
                    })}
                    onKeyDown={onKeyDown}
                    inputRef={(currentInputRef): void => {
                        inputRef.current = currentInputRef;
                    }}
                    id={randomName}
                    name={randomName}
                    value={input.value || ''}
                    placeholder={placeholder}
                    state={error ? 'error' : undefined}
                    innerRef={controlRef}
                    size="l"
                    disabled={disabled}
                    disabledAutocompleteFix
                    data-name={input.name}
                />
            )}
        />
    );
};

export default BookingInput;
