import React, {useCallback, useContext} from 'react';
import {FieldMetaState, useForm} from 'react-final-form';
import {OnChange} from 'react-final-form-listeners';

import {RU_CITIZENSHIP_CODE2} from 'constants/document/citizenship';
import {EFieldName} from '../../constants/fieldNames';
import {DOCUMENTS_WITH_CITIZEN} from '../../constants/documentTypes';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {TDocumentType} from 'types/common/document/TDocumentType';

import * as i18nBlock from 'i18n/components-BookingPassengerForm';

import Form from 'components/Form/Form';
import FormField from 'components/FormField/FormField';
import {TControlRenderFunc} from 'components/FormField/components/Field/Field';

interface IDocumentTypeFieldProps extends IWithClassName, IWithDeviceType {
    disabled?: boolean;
    initialDocumentType: TDocumentType;
    renderDocumentTypeSelect: TControlRenderFunc;
    excludeRuCitizenForOtherDocument?: boolean;
    onDocumentTypeChange: (documentType: TDocumentType) => void;
    onChangeError?: (fieldName: string, meta: FieldMetaState<string>) => void;
}

const DocumentTypeField: React.FC<IDocumentTypeFieldProps> = ({
    className,
    deviceType,
    disabled,
    renderDocumentTypeSelect,
    initialDocumentType,
    excludeRuCitizenForOtherDocument,
    onDocumentTypeChange,
    onChangeError,
}) => {
    const formApi = useForm();
    const parentGroupId = useContext(Form.FieldGroupContext);

    /* Событие ручного изменения типа документа */
    const handleDocumentManualChange = useCallback(() => {
        formApi.batch(() => {
            formApi.change(`${parentGroupId}.${EFieldName.documentNumber}`, '');
            formApi.change(
                `${parentGroupId}.${EFieldName.documentValidDate}`,
                '',
            );
        });
    }, [formApi, parentGroupId]);

    /* Событие любого изменения типа документа (вручную, интентом, саджестом) */
    const handleDocumentTypeChange = useCallback(
        (documentTypeValue: TDocumentType) => {
            formApi.batch(() => {
                if (!DOCUMENTS_WITH_CITIZEN.includes(documentTypeValue)) {
                    formApi.change(
                        `${parentGroupId}.${EFieldName.citizenship}`,
                        RU_CITIZENSHIP_CODE2,
                    );
                } else if (excludeRuCitizenForOtherDocument) {
                    formApi.change(
                        `${parentGroupId}.${EFieldName.citizenship}`,
                        null,
                    );
                }
            });

            onDocumentTypeChange(documentTypeValue);
        },
        [
            formApi,
            parentGroupId,
            onDocumentTypeChange,
            excludeRuCitizenForOtherDocument,
        ],
    );

    return (
        <>
            <FormField
                className={className}
                deviceType={deviceType}
                name={EFieldName.documentType}
                initialValue={initialDocumentType}
                title={i18nBlock.documentTypeDashTitle()}
                control={renderDocumentTypeSelect}
                disabled={disabled}
                onChange={handleDocumentManualChange}
                onChangeError={onChangeError}
            />
            <OnChange name={`${parentGroupId}.${EFieldName.documentType}`}>
                {handleDocumentTypeChange}
            </OnChange>
        </>
    );
};

export default DocumentTypeField;
