import React, {createContext, useMemo} from 'react';

import {ICountry} from 'types/common/ICountry';
import {TDocumentType} from 'types/common/document/TDocumentType';
import {IFormValidationInfo} from 'types/common/validation/form';
import {IPassengerWithDocumentsDTO} from 'server/api/TravelersApi/types/IPassengerDTO';
import {IIntentDocument, ISuggestDocument} from '../../types/IDocument';

import isNotEmptyDocument from './utilities/isNotEmptyDocument';
import flatMapDocuments from './utilities/flatMapDocuments';
import getDocumentsByPassengers from './utilities/getDocumentsByPassengers';
import omitDocumentsByParams from './utilities/omitDocumentsByParams';
import omitDocumentsByAvailability from './utilities/omitDocumentsByAvailability';
import omitDocumentsByValidation from 'components/BookingPassengerForm/components/DocumentsContext/utilities/omitDocumentsByValidation';

const MAX_INTENTS_COUNT = 10;
const DEFAULT_CONTEXT_DOCUMENTS = {
    suggestDocuments: [],
    intentDocuments: [],
};

interface IContextDocuments {
    suggestDocuments: ISuggestDocument[];
    intentDocuments: IIntentDocument[];
}

const DocumentsContext = createContext<IContextDocuments>(
    DEFAULT_CONTEXT_DOCUMENTS,
);

interface IDocumentsContextProps {
    passengers?: IPassengerWithDocumentsDTO[];
    countries: ICountry[];
    avaliableDocuments?: TDocumentType[];
    validationInfo?: IFormValidationInfo;
    excludeRuCitizenForOtherDocument?: boolean;
    hasPatronymic?: boolean;
    isOnlyName?: boolean;
    isLatinName?: boolean;
    /**
     * Не очищать ли поля documents от заведомо невалидных значений
     * @default false
     */
    keepInvalidFields?: boolean;
}

export const DocumentsContextProvider: React.FC<IDocumentsContextProps> = ({
    children,
    passengers,
    countries = [],
    avaliableDocuments,
    validationInfo,
    excludeRuCitizenForOtherDocument,
    hasPatronymic,
    isOnlyName,
    isLatinName,
    keepInvalidFields = false,
}) => {
    const contextDocuments: IContextDocuments = useMemo(() => {
        let documents = passengers
            ? getDocumentsByPassengers(passengers, countries, {
                  isLatinName,
                  hasPatronymic,
              })
            : [];

        documents = documents.map(docs =>
            omitDocumentsByParams(docs, {hasPatronymic, isOnlyName}),
        );

        documents = documents.map(docs =>
            omitDocumentsByAvailability(
                docs,
                avaliableDocuments,
                excludeRuCitizenForOtherDocument,
            ),
        );

        if (validationInfo && documents && !keepInvalidFields) {
            documents = documents.map(docs =>
                omitDocumentsByValidation(docs, validationInfo),
            );
        }

        const flatDocuments =
            flatMapDocuments(documents).filter(isNotEmptyDocument);

        const intentDocuments =
            flatDocuments.length > MAX_INTENTS_COUNT ? [] : flatDocuments;

        return {
            suggestDocuments: flatDocuments,
            intentDocuments,
        };
    }, [
        passengers,
        countries,
        validationInfo,
        excludeRuCitizenForOtherDocument,
        hasPatronymic,
        isOnlyName,
        isLatinName,
        avaliableDocuments,
    ]);

    return (
        <DocumentsContext.Provider value={contextDocuments}>
            {children}
        </DocumentsContext.Provider>
    );
};

export default DocumentsContext;
