import {
    RUSSIAN_PASSPORT,
    RUSSIAN_INTERNATIONAL_PASSPORT,
    BIRTH_CERTIFICATE,
    SAILOR_PASSPORT,
    MILITARY_CARD,
    DOCUMENT_OTHER,
} from 'constants/document/documentTypes';
import {EFieldName} from '../../../constants/fieldNames';
import {DOCUMENT_NAMES} from 'components/BookingPassengerForm/constants/documentTypes';

import {ICountry} from 'types/common/ICountry';
import {TDocumentType} from 'types/common/document/TDocumentType';
import {IDocumentDTO} from 'server/api/TravelersApi/types/IDocumentDTO';
import {IPassengerWithDocumentsDTO} from 'server/api/TravelersApi/types/IPassengerDTO';
import {IFormDocument, IFormDocumentMeta} from '../../../types/IDocument';
import {IContextDocument} from '../types/IContextDocument';

import {parseDate} from 'utilities/dateUtils';
import {ROBOT, HUMAN_DATE_RU} from 'utilities/dateUtils/formats';
import {translate} from 'utilities/strings/translate';

interface IMapDocumentParams {
    hasPatronymic?: boolean;
    isLatinName?: boolean;
}

/**
 * При выборе пассажира в интентах,
 * форма будет заполнена в соотвествии с приоритетом
 */
const DOCUMENT_TYPE_PRIORITY = {
    [RUSSIAN_PASSPORT]: 1,
    [BIRTH_CERTIFICATE]: 2,
    [RUSSIAN_INTERNATIONAL_PASSPORT]: 3,
    [DOCUMENT_OTHER]: 4,
    [SAILOR_PASSPORT]: 5,
    [MILITARY_CARD]: 6,
};

function getDocumentMeta(
    document: IDocumentDTO,
    params: IMapDocumentParams,
): IFormDocumentMeta {
    const isLatinName = Boolean(params.isLatinName);

    return {
        isTranslatedLastName:
            isLatinName &&
            !document.last_name_en &&
            Boolean(document.last_name),
        isTranslatedFirstName:
            isLatinName &&
            !document.first_name_en &&
            Boolean(document.first_name),
        isTranslatedPatronymicName:
            Boolean(params.hasPatronymic) &&
            isLatinName &&
            !document.middle_name_en &&
            Boolean(document.middle_name),
    };
}

function mapDocumentToForm(
    passenger: IPassengerWithDocumentsDTO,
    document: IDocumentDTO,
    countries: ICountry[],
    params: IMapDocumentParams,
): IFormDocument {
    const country = countries.find(c => c.geoId === document.citizenship);
    const lastName = params.isLatinName
        ? document.last_name_en || translate(document.last_name || '')
        : document.last_name || document.last_name_en;
    const firstName = params.isLatinName
        ? document.first_name_en || translate(document.first_name || '')
        : document.first_name || document.first_name_en;
    const patronymicName = params.isLatinName
        ? document.middle_name_en || translate(document.middle_name || '')
        : document.middle_name || document.middle_name_en;

    return {
        [EFieldName.lastName]: lastName || '',
        [EFieldName.firstName]: firstName || '',
        [EFieldName.patronymicName]: patronymicName,
        [EFieldName.sex]: passenger.gender,
        [EFieldName.birthdate]:
            passenger.birth_date &&
            parseDate(passenger.birth_date, ROBOT).format(HUMAN_DATE_RU),
        [EFieldName.documentType]: document.type,
        [EFieldName.documentNumber]: document.number,
        [EFieldName.citizenship]: country?.code2 || '',
        [EFieldName.documentValidDate]:
            document.expiration_date &&
            parseDate(document.expiration_date, ROBOT).format(HUMAN_DATE_RU),
    };
}

function getDocumentPriority(documentType?: TDocumentType): number {
    return documentType ? DOCUMENT_TYPE_PRIORITY[documentType] : 0;
}

function getDocumentComment(document: IFormDocument): string {
    const birthdate = document.birthdate || '';
    const documentName = document.documentType
        ? DOCUMENT_NAMES[document.documentType]
        : '';
    const birthdatePostfix = documentName && birthdate ? ', ' : '';

    return `${birthdate}${birthdatePostfix}${documentName}`;
}

/**
 * Возвращет массив массивов документов,
 * каждый подмассив содержит список документов одного пассажира
 */
function getDocumentsByPassengers(
    passengers: IPassengerWithDocumentsDTO[],
    countries: ICountry[],
    params: IMapDocumentParams,
): IContextDocument[][] {
    return passengers.map(passenger =>
        passenger.documents
            .map(document => {
                const meta = getDocumentMeta(document, params);
                const formDocument = mapDocumentToForm(
                    passenger,
                    document,
                    countries,
                    params,
                );

                return {
                    ...formDocument,
                    title: passenger.title,
                    text: passenger.title,
                    comment: getDocumentComment(formDocument),
                    meta,
                };
            })
            .sort(
                (firstDoc, secondDoc) =>
                    getDocumentPriority(firstDoc.documentType) -
                    getDocumentPriority(secondDoc.documentType),
            ),
    );
}

export default getDocumentsByPassengers;
