import _omit from 'lodash/omit';

import {EFieldName} from '../../../constants/fieldNames';
import {DOCUMENT_FIELDS} from '../../../constants/documentFields';

import {IFormValidationInfo} from 'types/common/validation/form';
import {IFormDocument} from '../../../types/IDocument';
import {IContextDocument} from '../types/IContextDocument';

import validateValues from 'utilities/validation/form/validateValues';

const MAIN_DOCUMENT_FIELDS = [
    EFieldName.documentType,
    EFieldName.documentValidDate,
];

/**
 * Удаляет невалидные свойства в документе.
 *
 * Если невалидны ключевые поля документа (MAIN_DOCUMENT_FIELDS),
 * то дополнительно удаляем остальные поля (DOCUMENT_FIELDS)
 */
export function omitDocumentByValidation<TDocument extends IFormDocument>(
    document: TDocument,
    validationInfo: IFormValidationInfo,
): TDocument {
    const errors = validateValues(
        {...validationInfo, isStrictStructure: false},
        document,
    );
    const errorFields = Object.keys(errors);
    const omitDocument = _omit(document, errorFields) as TDocument;

    const isOmitAllDocument = MAIN_DOCUMENT_FIELDS.some(field =>
        errorFields.includes(field),
    );

    if (isOmitAllDocument) {
        return _omit(omitDocument, DOCUMENT_FIELDS) as TDocument;
    }

    return omitDocument;
}

function omitDocumentsByValidation(
    documents: IContextDocument[],
    validationInfo: IFormValidationInfo,
): IContextDocument[] {
    return documents.reduce<IContextDocument[]>((docs, document) => {
        const omitDocument = omitDocumentByValidation(document, validationInfo);

        if (omitDocument) {
            docs.push(omitDocument);
        }

        return docs;
    }, []);
}

export default omitDocumentsByValidation;
