import React, {useCallback, useContext, useEffect} from 'react';

import {ECommonGoal} from 'utilities/metrika/types/goals/common';
import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {IIntentDocument} from '../../types/IDocument';

import {reachGoal} from 'utilities/metrika';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import IntentsList from 'components/IntentsList/IntentsList';

import DocumentsContext from '../DocumentsContext/DocumentsContext';

interface IPassengerIntentsListProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    onIntentSelect?: (document: IIntentDocument) => void;
}

const PassengerIntentsList: React.FC<IPassengerIntentsListProps> = ({
    className,
    deviceType,
    onIntentSelect,
    ...rest
}) => {
    const {intentDocuments} = useContext(DocumentsContext);

    const anyIntentDocument = Boolean(intentDocuments.length);

    useEffect(() => {
        if (anyIntentDocument) {
            reachGoal(ECommonGoal.NOTEBOOK_INTENTS_SHOW);
        }
    }, [anyIntentDocument]);

    const handleIntentClick = useCallback(
        (document: IIntentDocument) => {
            reachGoal(ECommonGoal.NOTEBOOK_INTENT_CLICK);
            onIntentSelect?.(document);
        },
        [onIntentSelect],
    );

    if (!anyIntentDocument) {
        return null;
    }

    return (
        <IntentsList
            className={className}
            items={intentDocuments}
            onIntentClick={handleIntentClick}
            nowrap={deviceType.isMobile}
            horizontalOffset={deviceType.isMobile ? 4 : 0}
            isCollapsed
            {...prepareQaAttributes(rest)}
        />
    );
};

export default PassengerIntentsList;
