import React, {ReactNode, useCallback, useContext} from 'react';
import {useForm} from 'react-final-form';
import {OnChange} from 'react-final-form-listeners';

import {EFieldName} from '../../constants/fieldNames';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {ISuggestDocument} from 'components/BookingPassengerForm/types/IDocument';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/components-BookingPassengerForm';

import Checkbox from 'components/Checkbox/Checkbox';
import Form, {useFieldValue} from 'components/Form/Form';

import SuggestInput from '../SuggestInput/SuggestInput';
import {IBookingInputProps} from '../BookingInput/BookingInput';

import cx from './PatronymicInput.scss';

interface IPatronymicInputProps extends IWithClassName, IWithDeviceType {
    bookInputProps: Omit<IBookingInputProps, 'onFocus' | 'onBlur' | 'name'> & {
        name: EFieldName;
    };
    suggestClassName?: string;
    initialCheckbox?: boolean;
    onSuggestSelect?: (document: ISuggestDocument) => void;
    renderOption?: (document: ISuggestDocument) => ReactNode;
}

const PatronymicInput: React.FC<IPatronymicInputProps> = ({
    className,
    suggestClassName,
    deviceType,
    initialCheckbox,
    bookInputProps,
    onSuggestSelect,
    renderOption,
}) => {
    const form = useForm();
    const formGroupId = useContext(Form.FieldGroupContext);
    const checkboxValue = useFieldValue<boolean>(
        EFieldName.isPatronymicDisabled,
    );

    const handleCheckboxChange = useCallback(
        (value: boolean) => {
            if (value) {
                form.change(`${formGroupId}.${bookInputProps.name}`, '');
            }
        },
        [form, formGroupId, bookInputProps.name],
    );

    return (
        <div className={className}>
            <SuggestInput
                className={suggestClassName}
                deviceType={deviceType}
                bookInputProps={{
                    ...bookInputProps,
                    disabled: bookInputProps.disabled || checkboxValue,
                    isHiddenError: checkboxValue,
                }}
                onSuggestSelect={onSuggestSelect}
                renderOption={renderOption}
            />

            <Form.Field
                name={EFieldName.isPatronymicDisabled}
                initialValue={initialCheckbox}
            >
                {({input}): React.ReactNode => (
                    <Checkbox
                        {...prepareQaAttributes({
                            current: `${bookInputProps.name}Checkbox`,
                            parent: bookInputProps,
                        })}
                        className={cx('checkbox')}
                        checked={input.value}
                        onChange={input.onChange}
                        onFocus={input.onFocus}
                        onBlur={input.onBlur}
                        disabled={bookInputProps.disabled}
                        size="m"
                        label={i18nBlock.patronymicDotNoPatronymicCheckbox()}
                    />
                )}
            </Form.Field>

            <OnChange
                name={`${formGroupId}.${EFieldName.isPatronymicDisabled}`}
            >
                {handleCheckboxChange}
            </OnChange>
        </div>
    );
};

export default PatronymicInput;
