import React, {
    useCallback,
    useContext,
    useMemo,
    useState,
    useEffect,
    ReactNode,
} from 'react';
import {useField} from 'react-final-form';
import {OnChange} from 'react-final-form-listeners';

import {EFieldName} from '../../constants/fieldNames';

import {ECommonGoal} from 'utilities/metrika/types/goals/common';
import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {ISuggestDocument} from '../../types/IDocument';

import {reachGoal} from 'utilities/metrika';
import getValidationErrorMessage from 'components/FormField/utilities/getValidationErrorMessage';
import filterDocumentsByValue from './utilities/filterDocumentsByValue';

import Form from 'components/Form/Form';
import Suggest, {
    ISuggestInputProps as ISuggestInputElementProps,
} from 'components/Suggest/Suggest';

import BookingInput, {IBookingInputProps} from '../BookingInput/BookingInput';
import DocumentsContext from '../DocumentsContext/DocumentsContext';

export interface ISuggestInputProps extends IWithClassName, IWithDeviceType {
    bookInputProps: Omit<IBookingInputProps, 'onFocus' | 'onBlur' | 'name'> & {
        name: EFieldName;
    };
    hint?: string;
    onSuggestSelect?: (document: ISuggestDocument) => void;
    renderOption?: (document: ISuggestDocument) => ReactNode;
}

/**
 * @description При автоматическом фокусе на ошибку не показывает suggest, начинает показывать только при изменении значения в поле
 */
const SuggestInput: React.FC<ISuggestInputProps> = ({
    className,
    deviceType,
    bookInputProps,
    hint,
    onSuggestSelect,
    renderOption,
}) => {
    const formGroupId = useContext(Form.FieldGroupContext);
    const {suggestDocuments} = useContext(DocumentsContext);
    const formInputName = `${formGroupId}.${bookInputProps.name}`;
    const [canShowSuggest, setCanShowSuggest] = useState(true);
    const {
        input: {value},
        meta,
    } = useField(formInputName);
    const submitAmount = meta.data?.submitAmount;

    const errorMessage = useMemo(
        () => getValidationErrorMessage(value, meta),
        [value, meta],
    );

    const filteredDocs = useMemo(
        () =>
            filterDocumentsByValue(
                suggestDocuments,
                bookInputProps.name,
                value,
            ),
        [suggestDocuments, bookInputProps.name, value],
    );

    // submitAmount используем как триггер события Submit
    useEffect(() => {
        setCanShowSuggest(!errorMessage);
    }, [submitAmount]);

    const handleInputChange = useCallback(() => setCanShowSuggest(true), []);

    const handleSuggestShow = useCallback(() => {
        reachGoal(ECommonGoal.NOTEBOOK_SUGGEST_SHOW);
    }, []);

    const handleElementSelect = useCallback(
        (document: ISuggestDocument) => {
            reachGoal(ECommonGoal.NOTEBOOK_SUGGEST_CLICK);
            onSuggestSelect?.(document);
        },
        [onSuggestSelect],
    );

    const renderInput = useCallback(
        (inputProps: ISuggestInputElementProps) => (
            <BookingInput
                {...bookInputProps}
                hint={hint}
                onKeyDown={inputProps.onKeyDown}
                onChange={inputProps.onChange}
                onFocus={inputProps.onFocus}
                onBlur={inputProps.onBlur}
                inputRef={inputProps.inputRef}
            />
        ),
        [bookInputProps, hint],
    );

    return (
        <>
            <Suggest
                className={className}
                deviceType={deviceType}
                onOptionSelect={handleElementSelect}
                options={canShowSuggest ? filteredDocs : []}
                onShowPopup={handleSuggestShow}
                preventEnter
                renderOption={renderOption}
                renderInput={renderInput}
            />
            {/*
                Не используется onInputChange компонента Suggest,
                т.к. изменение значений формы через FormApi не вызывает обработчик onChange
            */}
            <OnChange name={formInputName}>{handleInputChange}</OnChange>
        </>
    );
};

export default SuggestInput;
