import {EFieldName} from '../../../constants/fieldNames';

import {ISuggestDocument} from '../../../types/IDocument';

const MAX_SUGGEST_COUNT = 5;

function checkLastName(
    document: ISuggestDocument,
    inputValue: string,
): boolean {
    return document.lastName?.toLowerCase().includes(inputValue) || false;
}

function checkFirstName(
    document: ISuggestDocument,
    inputValue: string,
): boolean {
    return document.firstName?.toLowerCase().includes(inputValue) || false;
}

function checkMiddleName(
    document: ISuggestDocument,
    inputValue: string,
): boolean {
    return document.patronymicName?.toLowerCase().includes(inputValue) || false;
}

function checkTitle(document: ISuggestDocument, inputValue: string): boolean {
    return document.title?.toLowerCase().includes(inputValue) || false;
}

function checkFieldDocuments(
    document: ISuggestDocument,
    fieldName: EFieldName,
    inputValue: string,
): boolean {
    const caseValue = inputValue.toLowerCase();

    switch (fieldName) {
        case EFieldName.lastName:
            return checkLastName(document, caseValue);
        case EFieldName.firstName:
            return checkFirstName(document, caseValue);
        case EFieldName.patronymicName:
            return checkMiddleName(document, caseValue);
    }

    return false;
}

function checkOtherDocuments(
    document: ISuggestDocument,
    fieldName: EFieldName,
    inputValue: string,
): boolean {
    const caseValue = inputValue.toLowerCase();

    switch (fieldName) {
        case EFieldName.lastName:
            return (
                checkFirstName(document, caseValue) ||
                checkMiddleName(document, caseValue) ||
                checkTitle(document, caseValue)
            );
        case EFieldName.firstName:
            return (
                checkLastName(document, caseValue) ||
                checkMiddleName(document, caseValue) ||
                checkTitle(document, caseValue)
            );
        case EFieldName.patronymicName:
            return (
                checkLastName(document, caseValue) ||
                checkFirstName(document, caseValue) ||
                checkTitle(document, caseValue)
            );
    }

    return false;
}

/**
 * Фильтрует список документов сначала по значению поля fieldName,
 * потом по значениям остальных полей
 */
function filterDocumentsByValue(
    documents: ISuggestDocument[],
    fieldName: EFieldName,
    inputValue?: string,
): ISuggestDocument[] {
    if (!inputValue) {
        return documents.slice(0, MAX_SUGGEST_COUNT);
    }

    const documentsByField = documents.filter(document =>
        checkFieldDocuments(document, fieldName, inputValue),
    );

    if (documentsByField.length >= MAX_SUGGEST_COUNT) {
        return documentsByField.slice(0, MAX_SUGGEST_COUNT);
    }

    const documentsByOthersFields = documents.filter(
        document =>
            !documentsByField.includes(document) &&
            checkOtherDocuments(document, fieldName, inputValue),
    );

    return documentsByField
        .concat(documentsByOthersFields)
        .slice(0, MAX_SUGGEST_COUNT);
}

export default filterDocumentsByValue;
