import {FormApi} from 'final-form';

import {EFieldName} from '../constants/fieldNames';
import {
    DOCUMENTS_WITH_CITIZEN,
    DOCUMENTS_WITH_DATE,
} from '../constants/documentTypes';
import {DOCUMENT_FIELDS} from '../constants/documentFields';

import {TDocumentType} from 'types/common/document/TDocumentType';
import {IFormDocument} from '../types/IDocument';

interface ISetFormValuesOptions {
    hasPatronymic?: boolean;
    isOnlyName?: boolean;
}

function resetDocumentValues(form: FormApi, formGroupId: string): void {
    DOCUMENT_FIELDS.forEach(documentField => {
        const fullFieldName = `${formGroupId}.${documentField}`;
        const initalValue = form.getFieldState(fullFieldName)?.initial;

        form.change(fullFieldName, initalValue);
    });
}

function setDocumentValues(
    form: FormApi,
    formGroupId: string,
    document: IFormDocument,
): void {
    const {documentType, documentNumber, citizenship, documentValidDate} =
        document;

    form.change(`${formGroupId}.${EFieldName.documentType}`, documentType);
    form.change(`${formGroupId}.${EFieldName.documentNumber}`, documentNumber);

    // Неочевидный момент: На проде в ЗКП оказался паспорт РФ со сроком годности
    // поэтому не доверяем до конца данным и делаем проверку на тип документа
    if (documentType && DOCUMENTS_WITH_CITIZEN.includes(documentType)) {
        form.change(`${formGroupId}.${EFieldName.citizenship}`, citizenship);
    }

    const isDocumentValidDateSet =
        documentValidDate &&
        documentType &&
        DOCUMENTS_WITH_DATE.includes(documentType);

    form.change(
        `${formGroupId}.${EFieldName.documentValidDate}`,
        isDocumentValidDateSet ? documentValidDate : '',
    );
}

function setFormValuesByDocument(
    form: FormApi,
    formGroupId: string,
    document: IFormDocument,
    avaliableDocuments: TDocumentType[] | undefined,
    options: ISetFormValuesOptions,
): void {
    const {hasPatronymic, isOnlyName} = options;
    const {lastName, firstName, patronymicName, sex, birthdate, documentType} =
        document;

    const documentTypeIsAvaliable = avaliableDocuments
        ? documentType && avaliableDocuments.includes(documentType)
        : Boolean(documentType);

    form.batch(() => {
        form.change(`${formGroupId}.${EFieldName.lastName}`, lastName);
        form.change(`${formGroupId}.${EFieldName.firstName}`, firstName);

        if (hasPatronymic) {
            form.change(
                `${formGroupId}.${EFieldName.isPatronymicDisabled}`,
                !patronymicName,
            );

            form.change(
                `${formGroupId}.${EFieldName.patronymicName}`,
                patronymicName,
            );
        }

        if (!isOnlyName) {
            form.change(`${formGroupId}.${EFieldName.sex}`, sex);
            form.change(`${formGroupId}.${EFieldName.birthdate}`, birthdate);
        }

        if (!isOnlyName && documentTypeIsAvaliable) {
            setDocumentValues(form, formGroupId, document);
        } else {
            resetDocumentValues(form, formGroupId);
        }
    });
}

export default setFormValuesByDocument;
