import {FC, ReactNode} from 'react';

import {EFieldName} from './constants/fieldNames';

import {IWithClassName} from 'types/withClassName';
import {IFieldError} from 'types/common/validation/form';

import {deviceModMobile} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useWhiteLabelConfig} from 'utilities/hooks/useWhiteLabelConfig';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import Text from 'components/Text/Text';
import Input from 'components/Input/Input';
import FormField from 'components/FormField/FormField';
import Flex from 'components/Flex/Flex';

import cx from './BookingWhiteLabelForm.scss';

export type TBookingWhiteLabel = Partial<Record<EFieldName, string>>;

export type TBookingWhiteLabelErrors = Partial<Record<EFieldName, IFieldError>>;

export interface IBookingWhiteLabelFormProps
    extends IWithQaAttributes,
        IWithClassName {}

const BookingWhiteLabelForm: FC<IBookingWhiteLabelFormProps> = props => {
    const {className} = props;
    const deviceType = useDeviceType();
    const {isDesktop} = deviceType;
    const {config: whiteLabelConfig} = useWhiteLabelConfig();
    const logo = whiteLabelConfig?.loyaltyLogo;
    const title = whiteLabelConfig?.orderCustomerNumberTitle;
    const description = whiteLabelConfig?.orderCustomerNumberDescription;

    return (
        <fieldset className={cx('root', className)}>
            {logo && isDesktop && (
                <img src={logo} alt="" className={cx('logo')} />
            )}

            <div {...prepareQaAttributes(props)}>
                <Text
                    className={cx(
                        'title',
                        deviceModMobile('title', deviceType),
                    )}
                    size="l"
                    tag="legend"
                    weight="bold"
                >
                    {title}
                </Text>

                <FormField
                    name={EFieldName.customerNumber}
                    title={
                        <div className={cx('description')}>{description}</div>
                    }
                    deviceType={deviceType}
                    control={(
                        {input},
                        {error, inputRef, controlRef},
                    ): ReactNode => (
                        <Flex>
                            {logo && !isDesktop && (
                                <img
                                    src={logo}
                                    alt=""
                                    className={cx(
                                        'logo',
                                        deviceModMobile('logo', deviceType),
                                    )}
                                />
                            )}
                            <Input
                                {...input}
                                className={cx(
                                    'input',
                                    deviceModMobile('input', deviceType),
                                )}
                                id={input.name}
                                inputRef={(currentInputRef): void => {
                                    inputRef.current = currentInputRef;
                                }}
                                innerRef={controlRef}
                                value={input.value || null}
                                size="l"
                                state={error ? 'error' : undefined}
                                type="text"
                                inputMode="text"
                                disabledAutocompleteFix
                            />
                        </Flex>
                    )}
                />
            </div>
        </fieldset>
    );
};

export default BookingWhiteLabelForm;
