import React, {RefObject, useCallback, useState} from 'react';

import {IWithClassName} from 'types/withClassName';

import useSupportHistoryBack from 'utilities/hooks/useSupportHistoryBack';
import {useLayer} from 'utilities/hooks/useLayer';
import {useEventListener} from 'utilities/hooks/useEventListener';

import {Drawer} from '@yandex-lego/components/Drawer/touch-phone/bundle';
import {MODAL_LAYER_MULTIPLIER} from 'components/Modal/Modal';
import BottomSheetContent from './components/BottomSheetContent/BottomSheetContent';
import CloseIcon from 'icons/16/Close';
import useBottomSheetContext from './components/BottomSheetManager/useBottomSheetManager';

import LayersContext from 'contexts/LayersContext';

import usePreventScrollFix from './hooks/usePreventScrollFix';

import cx from './BottomSheet.scss';

const animation = {tension: 230, friction: 24};

export interface IBottomSheetProps extends IWithClassName {
    isOpened: boolean;
    onClose?(
        e?:
            | React.MouseEvent<HTMLButtonElement, MouseEvent>
            | React.MouseEvent<SVGElement, MouseEvent>
            | Event,
    ): void;
    /**
     * Признак вложенности шторки.
     * В этом случае "язычок" шторки будет не над ней, а внутри нее.
     */
    nested?: boolean;
    /**
     * Максимальный размер шторки.
     * По сути это то, что будет в свойстве max-height шторки.
     * @default 'calc(100% - 48px)'
     */
    maxSize?: string;
    /**
     * Убирает скролл контента шторки по горизонтали.
     * @default false
     */
    withoutHorizontalScroll?: boolean;
    popHistoryOnUnmount?: boolean;
    contentClassName?: string;
    dragDisabledContent?: boolean;
    hasCloseButton?: boolean;
    paddingTopSpace?: 0;
    paddingBottomSpace?: 0;
    padding?: 0;
    innerRef?: RefObject<HTMLDivElement>;
    keepMounted?: boolean;
}

const BottomSheet: React.FC<IBottomSheetProps> = props => {
    const {
        className,
        contentClassName,
        isOpened,
        children,
        onClose,
        popHistoryOnUnmount = true,
        nested = false,
        maxSize = 'calc(100% - 48px)',
        withoutHorizontalScroll = false,
        dragDisabledContent,
        hasCloseButton,
        paddingTopSpace,
        paddingBottomSpace,
        innerRef,
        keepMounted,
        padding,
    } = props;

    useSupportHistoryBack({
        isVisible: isOpened,
        close: onClose,
        popHistoryOnUnmount,
    });

    const [key, setKey] = useState(0);

    const handleOrientationChange = useCallback(() => {
        setKey(key + 1);
    }, [setKey, key]);

    useEventListener('orientationchange', handleOrientationChange);

    const layer = useLayer();

    const {dragDisabled} = useBottomSheetContext({layer, isOpened});

    usePreventScrollFix();

    if (!children || typeof document === 'undefined') {
        return null;
    }

    return (
        <Drawer
            className={cx('root', className, {
                root_closed: !isOpened,
            })}
            animation={animation}
            view="default"
            key={key}
            visible={isOpened}
            onClose={onClose}
            nested={nested}
            maxSize={maxSize}
            keepMounted={keepMounted}
            zIndex={layer * MODAL_LAYER_MULTIPLIER} // Такой z-index связан с тем, что именно так он задается в компоненте Modal и в случае, когда BottomSheet находится внутри Modal приходится выставлять x-index больше, чем у Modal
            dragDisabled={dragDisabled}
        >
            <LayersContext.Provider value={layer}>
                <BottomSheetContent
                    className={contentClassName}
                    dragDisabledContent={dragDisabledContent}
                    paddingTopSpace={paddingTopSpace}
                    paddingBottomSpace={paddingBottomSpace}
                    innerRef={innerRef}
                    padding={padding}
                    withoutHorizontalScroll={withoutHorizontalScroll}
                >
                    {hasCloseButton && (
                        <CloseIcon className={cx('close')} onClick={onClose} />
                    )}
                    {children}
                </BottomSheetContent>
            </LayersContext.Provider>
        </Drawer>
    );
};

export default BottomSheet;
