import React, {RefObject, useCallback, createRef} from 'react';

import {IOS} from 'utilities/deviceType/constants';

import {IWithClassName} from 'types/withClassName';

import {isOS} from 'utilities/deviceType/isOS';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useEventListener} from 'utilities/hooks/useEventListener';

import cx from './BottomSheetContent.scss';

export interface IBottomSheetContentProps extends IWithClassName {
    dragDisabledContent?: boolean;
    paddingTopSpace?: 0;
    paddingBottomSpace?: 0;
    padding?: 0;
    innerRef?: RefObject<HTMLDivElement>;
    /**
     * Убирает скролл контента шторки по горизонтали.
     * @default false
     */
    withoutHorizontalScroll?: boolean;
}

const cancelEventOptions: AddEventListenerOptions = {
    passive: true,
};

const BottomSheetContent: React.FC<IBottomSheetContentProps> = props => {
    const {
        className,
        children,
        dragDisabledContent,
        paddingTopSpace,
        paddingBottomSpace,
        padding,
        innerRef,
        withoutHorizontalScroll = false,
    } = props;

    const {os} = useDeviceType();
    const contentRef = innerRef || createRef<HTMLDivElement>();
    const cancelEvent = useCallback(
        event => {
            if (!dragDisabledContent) {
                return;
            }

            event.stopPropagation();
        },
        [dragDisabledContent],
    );

    useEventListener('touchmove', cancelEvent, contentRef, cancelEventOptions);

    return (
        <div
            className={cx(
                'content',
                className,
                `content_paddingTopSpace_${paddingTopSpace}`,
                `content_paddingBottomSpace_${paddingBottomSpace}`,
                `content_paddingSpace_${padding}`,
                {
                    content_IOS: isOS(os, IOS),
                    content_withoutHorizontalScroll: withoutHorizontalScroll,
                },
            )}
            ref={contentRef}
        >
            {children}
        </div>
    );
};

export default BottomSheetContent;
