import {Meta} from '@storybook/react/types-6-0';

import {TStorybookFunctionComponent} from 'storybook/types';

import Box from 'components/Box/Box';
import Text from 'components/Text/Text';
import StyleguidistSizeTable from 'components/StyleguidistSizeTable/StyleguidistSizeTable';
import StyleguidistPropTable from 'components/StyleguidistPropTable/StyleguidistPropTable';
import DownloadIcon from 'icons/16/Download';

import Button, {TButtonTheme, TButtonSize} from './Button';
import {storyButtons, Td} from './__docs__/storyWithIcons';

export default {
    title: 'Example/Button',
    component: Button,
} as Meta;

const THEMES: Record<TButtonTheme, string> = {
    secondary: 'Secondary',
    primary: 'Primary',
    outlined: 'Outlined',
    attention: 'Attention',
    raised: 'Raised',
    plain: 'Plain',
    plus: 'Plus',
};

const SIZES: Record<TButtonSize, string> = {
    s: '28px',
    'm-inset': '36px',
    m: '36px',
    l: '44px',
    xl: '52px',
};

export const Themes: TStorybookFunctionComponent = () => (
    <StyleguidistPropTable
        props={Object.entries(THEMES).map(([theme, name]) => ({
            name,
            props: {theme},
            theme: theme === 'plain' ? 'secondary' : undefined,
            example: (
                <Box between="5" inline>
                    <Button theme={theme as TButtonTheme}>Кнопушка</Button>
                    <Button theme={theme as TButtonTheme} disabled>
                        Кнопушка
                    </Button>
                </Box>
            ),
        }))}
    />
);

export const Shapes: TStorybookFunctionComponent = () => (
    <StyleguidistPropTable
        props={[
            {
                name: 'Round',
                props: {},
                example: <Button size="l">-</Button>,
                comment: 'default',
            },
            {
                name: 'Circle',
                props: {shape: 'circle'},
                example: (
                    <Button size="l" shape="circle">
                        +
                    </Button>
                ),
            },
        ]}
    />
);

export const Sizes: TStorybookFunctionComponent = () => (
    <StyleguidistSizeTable
        sizes={Object.entries(SIZES).map(([name, size]) => ({
            name,
            size,
            example: <Button size={name as TButtonSize}>Кнопуля</Button>,
        }))}
    />
);

export const Widths: TStorybookFunctionComponent = () => (
    <Box between="2">
        <Box>
            <Button>Auto</Button>
        </Box>
        <Box>
            <Button width="max">Max</Button>
        </Box>
        <Box>
            <Button width="max" iconRight={<DownloadIcon />}>
                Max with Icon
            </Button>
        </Box>
    </Box>
);

export const WithIcons: TStorybookFunctionComponent = () => (
    <table>
        <tbody>
            {storyButtons.map(({size, theme, text, icon}) => (
                <tr key={size}>
                    <Td>
                        <Text size="l" weight="bold">
                            {size}
                        </Text>
                    </Td>
                    <Td>
                        <Button icon={icon} size={size} theme={theme}>
                            {text}
                        </Button>
                    </Td>
                    <Td>
                        <Button iconRight={icon} size={size} theme={theme}>
                            {text}
                        </Button>
                    </Td>
                    <Td>
                        <Button
                            iconLeft={icon}
                            iconRight={icon}
                            size={size}
                            theme={theme}
                        >
                            {text}
                        </Button>
                    </Td>
                    <Td>
                        <Button iconLeft={icon} size={size} theme={theme} />
                    </Td>
                </tr>
            ))}
        </tbody>
    </table>
);
