import React, {
    MutableRefObject,
    useCallback,
    useMemo,
    useRef,
    useState,
} from 'react';
import {noop} from 'lodash';

import {IWithClassName} from 'types/withClassName';
import {TCalendarPrices} from 'types/common/calendarPrice/ICalendarPrice';
import {EDatePickerSelectionMode} from 'components/DatePicker/types';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import createCalendarMonths from 'components/Calendar/utilities/createCalendarMonths';
import createCalendarWeekdays from 'components/Calendar/utilities/createCalendarWeekdays';
import {getMonthListYears} from 'components/Calendar/utilities/getMonthListYears';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import CalendarMonthsGrid from 'components/Calendar/components/CalendarMonthsGrid/CalendarMonthsGridNew';
import CalendarMonthsList from 'components/Calendar/components/CalendarMonthsList/CalendarMonthsListNew';
import Select from 'components/Select/Select';
import Flex from 'components/Flex/Flex';
import ArrowBottomIcon from 'icons/16/ArrowBottom';

import cx from './Calendar.scss';

export interface ICalendarProps extends IWithClassName, IWithQaAttributes {
    calendarWrapperClassName?: string;
    dayClassName?: string;
    weekdaysClassName?: string;
    monthsGridClassName?: string;

    minDate: Date;
    maxDate: Date;
    nowDate: Date;
    startDate?: Nullable<Date>;
    endDate?: Nullable<Date>;
    weekStart?: number;
    siblingMonths?: boolean;
    footerNode?: React.ReactNode;
    headerNode?: React.ReactNode;
    isRenderMonthsList: boolean;
    cutPastWeeks?: boolean;
    canScrollIntoSelectedDate?: boolean;
    prices?: TCalendarPrices;
    canSelectRange?: boolean;
    selectionMode: EDatePickerSelectionMode;
    yearSelectRef: MutableRefObject<HTMLElement | null>;
    withYearSelect?: boolean;

    onDaySelect?: (data: {selectedDate: Date; dayNode: HTMLDivElement}) => void;
    onDayHover?: (date: Nullable<Date>) => void;

    onMonthsGridTouchStart?: (event: TouchEvent) => void;
    onMonthsGridTouchMove?: (event: TouchEvent) => void;
    onMonthsGridTouchEnd?: (event: TouchEvent) => void;
}

const CalendarNew: React.FC<ICalendarProps> = props => {
    const {
        className,
        calendarWrapperClassName,
        dayClassName,
        weekdaysClassName,
        monthsGridClassName,
        prices,
        minDate,
        maxDate,
        nowDate,
        onDaySelect = noop,
        startDate = null,
        endDate = null,
        weekStart = 1,
        siblingMonths = false,
        footerNode = null,
        headerNode = null,
        isRenderMonthsList = true,
        cutPastWeeks = false,
        canScrollIntoSelectedDate = true,
        onMonthsGridTouchEnd,
        onMonthsGridTouchStart,
        onMonthsGridTouchMove,
        onDayHover,
        canSelectRange = false,
        selectionMode,
        yearSelectRef,
        withYearSelect,
    } = props;
    const deviceType = useDeviceType();

    const [hoveredDate, setHoveredDate] = useState<null | Date>(null);
    const handleDayClick = useCallback(
        (selectedDate: Date) => {
            onDaySelect({
                selectedDate,
            });
        },
        [onDaySelect],
    );

    const handleSetHoveredDate = useCallback(
        (newValue: Nullable<Date>) => {
            if (deviceType.isMobile) {
                return;
            }

            setHoveredDate(newValue);
            onDayHover?.(newValue);
        },
        [onDayHover, setHoveredDate, deviceType.isMobile],
    );

    const months = useMemo(
        () =>
            createCalendarMonths({
                minDate,
                maxDate,
                weekStart,
                siblingMonths,
                cutPastWeeks,
                cutDate: nowDate,
            }),
        [minDate, maxDate, weekStart, siblingMonths, cutPastWeeks, nowDate],
    );
    const weekdays = useMemo(
        () => createCalendarWeekdays({weekStart}),
        [weekStart],
    );

    const isRangeSelected = useMemo(() => {
        if (!startDate || !canSelectRange) {
            return false;
        }

        if (endDate && Number(startDate) !== Number(endDate)) {
            return true;
        }

        return (
            hoveredDate !== null && Number(hoveredDate) !== Number(startDate)
        );
    }, [startDate, endDate, hoveredDate, canSelectRange]);

    const [visibleMonth, setVisibleMonth] = useState({
        index: 0,
        month: months[0],
    });
    const monthsGridRef = useRef<CalendarMonthsGrid>(null);
    const monthsListRef = useRef<CalendarMonthsList>(null);
    const scrollToMonth = useCallback((monthIndex: number) => {
        if (!monthsGridRef.current) {
            return;
        }

        monthsGridRef.current.scrollToDate({monthIndex});
    }, []);

    const handleYearSelect = useCallback(
        year => {
            for (let i = 0; i < months.length; i++) {
                const month = months[i];

                if (
                    month.year === year &&
                    month.month === visibleMonth.month.month
                ) {
                    const januaryIndex = i - months[i].month;

                    monthsListRef.current?.scrollToMonth(januaryIndex);
                    setVisibleMonth({
                        index: i,
                        month: months[i],
                    });
                    scrollToMonth(i);

                    return;
                }
            }
        },
        [setVisibleMonth, scrollToMonth, months, visibleMonth],
    );

    const years = useMemo(
        () =>
            getMonthListYears(months).map(value => ({
                value,
                data: value,
            })),
        [months],
    );

    return (
        <div
            className={cx(
                'calendar',
                className,
                deviceMods('calendar', deviceType),
            )}
            {...prepareQaAttributes(props)}
        >
            <Flex className={cx('yearAndMonth')} flexDirection="column">
                {withYearSelect && (
                    <Select
                        options={years}
                        onChange={handleYearSelect}
                        icon={ArrowBottomIcon}
                        theme="plain"
                        size="m"
                        value={visibleMonth.month.year}
                        className={cx('year')}
                        listRef={yearSelectRef}
                    />
                )}
                {isRenderMonthsList && (
                    <CalendarMonthsList
                        className={cx('monthsList', 'monthsList_new')}
                        months={months}
                        scrollToMonth={scrollToMonth}
                        highlightMonth={visibleMonth.index}
                        ref={monthsListRef}
                        canSelectRange={canSelectRange}
                        startDate={startDate ? startDate : undefined}
                        endDate={endDate ? endDate : undefined}
                        nowDate={nowDate}
                        minDate={minDate}
                        maxDate={maxDate}
                        hoveredDate={hoveredDate ? hoveredDate : undefined}
                    />
                )}
            </Flex>
            <div className={cx('calendarWrapper', calendarWrapperClassName)}>
                {headerNode && <div className={cx('header')}>{headerNode}</div>}
                <CalendarMonthsGrid
                    className={cx('monthsGrid', monthsGridClassName)}
                    ref={monthsGridRef}
                    dayClassName={dayClassName}
                    weekdaysClassName={weekdaysClassName}
                    months={months}
                    weekdays={weekdays}
                    canSelectRange={canSelectRange}
                    startDate={startDate ? startDate : undefined}
                    endDate={endDate ? endDate : undefined}
                    nowDate={nowDate}
                    minDate={minDate}
                    maxDate={maxDate}
                    isRangeSelected={Boolean(isRangeSelected)}
                    deviceType={deviceType}
                    hoveredDate={hoveredDate ? hoveredDate : undefined}
                    onVisibleMonthChange={setVisibleMonth}
                    canScrollIntoSelectedDate={canScrollIntoSelectedDate}
                    prices={prices}
                    selectionMode={selectionMode}
                    onDayClick={handleDayClick}
                    setHoveredDate={handleSetHoveredDate}
                    onTouchStart={onMonthsGridTouchStart}
                    onTouchMove={onMonthsGridTouchMove}
                    onTouchEnd={onMonthsGridTouchEnd}
                    {...prepareQaAttributes(props)}
                />
                {footerNode}
            </div>
        </div>
    );
};

export default CalendarNew;
